import logging
import subprocess
import os

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.enum import Enum

from sandbox.projects.quasar.resource_types import (
    QuasarGstreamer
)

from sandbox.projects.quasar.utils import (
    SafePublishingMixing
)

from sandbox.sdk2.vcs.git import Git
from sandbox.common.types import misc

SETUP_DOCKER = """
#!/usr/bin/env bash

set -x

export DEBIAN_FRONTEND=noninteractive

echo "# Path to docker root: $1"

sudo apt-get update
sudo apt-get install --yes --force-yes apt-transport-https ca-certificates curl software-properties-common

dpkg -L software-properties-common

echo $PATH

which add-apt-repository

curl -fsSL https://download.docker.com/linux/ubuntu/gpg | apt-key add -
apt-key fingerprint 0EBFCD88
add-apt-repository "deb [arch=amd64] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable"

apt-get update
apt-get install -y --force-yes docker-ce=17.09.1* unbound-config-local64

default_gw_if=$(ip -6 r s | grep -m 1 ^default | awk '{print $5}')

ip -6 r s
ip a

mkdir -p "$1"
service docker stop
ip link del docker0
rm -rf /var/lib/docker

echo > /etc/default/docker
echo 'DOCKER_OPTS="-D -s overlay -H unix://'"$1"'/docker.sock --ip-forward=false --ipv6 \
    --fixed-cidr-v6 fd00::/8 \
    --graph '"$1"'"' | tee -a /etc/default/docker

service docker start

ip -6 r s
ip a

ip6tables -t nat -A POSTROUTING -s fd00::/8 -j MASQUERADE
echo 2 > /proc/sys/net/ipv6/conf/${default_gw_if}/accept_ra
echo 1 > /proc/sys/net/ipv6/conf/all/forwarding

ip -6 r s
ip a

docker -H unix://$1/docker.sock info

set +o xtrace
docker -H unix://$1/docker.sock login -u $2 -p $3 registry.yandex.net
"""


class GstPlatform(Enum):
    Enum.preserve_order()
    Enum.lower_case()

    ANDROID = None
    LINKPLAY_A98 = None
    X86_64 = None
    WK7Y = None


class QuasarGstreamerBuild(SafePublishingMixing, sdk2.Task):
    """
    Build gstreamer for quasar
    """

    GIT_URL = 'https://github.yandex-team.ru/quasar-dev/io-external-libs'
    ARCADIA_DIR = 'arcadia'
    CHECKOUT_PATH = os.path.join(ARCADIA_DIR, "quasar", "io-external-libs")

    VAULT_OWNER = 'QUASAR'
    DOCKER_REGISTRY_AUTH_TOKEN = 'quasar_docker_registry_auth_token'
    DOCKER_REGISTRY_AUTH_LOGIN = 'quasar_docker_registry_auth_login'

    class Requirements(sdk2.Requirements):
        privileged = True
        dns = misc.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=876418497,
            required=True)

        branch = sdk2.parameters.String('Git io-external-libs branch', default='master', required=True)

        all_platforms = [platform for platform in list(GstPlatform)]
        platforms_to_build = sdk2.parameters.String(
            'Comma separated list of platforms to build for. Values should be one of [{}].'.format(
                ', '.join(all_platforms)),
            default=', '.join(all_platforms))

        rebuild_image = sdk2.parameters.Bool('Rebuild docker image', default=True)
        publish_resources = sdk2.parameters.Bool('Publish resources', default=True)
        use_host_network = sdk2.parameters.Bool('Use host network for docker', default=True)

        with sdk2.parameters.Output:
            output_resource = sdk2.parameters.Resource('Libs resource id', required=False)

    def on_execute(self):
        docker_setup_script = os.path.join(str(self.path()), 'docker_setup.sh')
        logging.info('Docker setup script: {}'.format(docker_setup_script))
        with open(docker_setup_script, 'w') as fscript:
            fscript.write(SETUP_DOCKER)

        with sdk2.helpers.ProcessLog(self, logger="docker_prepare") as pl:
            self.docker_root = os.path.join(str(self.path()), 'docker_root')
            registry_login = sdk2.Vault.data(self.VAULT_OWNER, self.DOCKER_REGISTRY_AUTH_LOGIN)
            registry_token = sdk2.Vault.data(self.VAULT_OWNER, self.DOCKER_REGISTRY_AUTH_TOKEN)
            retcode = subprocess.Popen("bash {} {} {} {}".format(docker_setup_script, self.docker_root, registry_login, registry_token), shell=True, stderr=pl.stdout, stdout=pl.stdout).wait()
            if retcode == 0:
                logging.info("Docker prepared successfully")
            else:
                self.docker_root = ""
                raise Exception("Docker prepare failed")

        os.remove(docker_setup_script)

        if self.Parameters.use_host_network:
            os.environ["HOST_NETWORK"] = "True"

        os.environ["DOCKER_SOCK"] = self.docker_sock()

        if self.Parameters.rebuild_image:
            logging.info('rebuilding image')
            self.rebuild_image()

        platforms = set(platform.strip() for platform in self.Parameters.platforms_to_build.split(','))
        if not platforms.issubset(set(GstPlatform)):
            raise Exception('Platforms {} are not allowed'.format(','.join(platforms - set(GstPlatform))))

        for platform in platforms:
            logging.info('building alsa-lib,cerbero for {}'.format(platform))
            self.build_lib(platform, "alsa-lib-1.1.3,cerbero-1.14.3")

            published = self.publish(
                {QuasarGstreamer: 'build/local/cerbero-1.14.3/lib/{}'.format(platform)},
                platform=platform)

            if len(platforms) == 1 and published:
                self.Parameters.output_resource = published.values()[0].id
            else:
                logging.info("Output resource parameter is not filled if multiple output platforms are specified")

    def on_prepare(self):
        git = Git(self.GIT_URL)
        git.clone(self.CHECKOUT_PATH, self.Parameters.branch)
        git.execute("submodule", "update", "--init", "--recursive", cwd=self.CHECKOUT_PATH)

    def rebuild_image(self):
        exit_code = self.run_build_script("image")
        if exit_code != 0:
            raise errors.TaskFailure('Docker image rebuild exited with {}'.format(exit_code))

    def build_lib(self, platform, lib):
        os.environ["LIB"] = lib

        exit_code = self.run_build_script(platform)
        if exit_code != 0:
            raise errors.TaskFailure('{} build for platform {} exited with {}'.format(lib, platform, exit_code))

    def run_build_script(self, args):
        with sdk2.helpers.ProcessLog(self, logger="script-log") as pl:
            exit_code = subprocess.call(["/bin/bash", "-x", "-c", "./build.sh {}".format(args)],
                                        cwd=self.CHECKOUT_PATH,
                                        stdout=pl.stdout,
                                        stderr=pl.stdout)
            return exit_code

    def docker_sock(self):
        if self.docker_root == "":
            raise Exception("No docker environment prepared")

        return "unix://{}".format(os.path.join(self.docker_root, 'docker.sock'))

    def publish(self, resources, **resources_attrs):
        """
        Publishes resource.

        To make sure that they are not removed later the resource is first copied
        to some "safe" dir in the task home.

        :param resources:
            dict {resource_class: 'path/to/resources'} to be published

        :param **resources_attrs:
            extra attributes for resources to be assigned
        """

        if not self.Parameters.publish_resources:
            logging.info('Skipping publish because parameter publish_resources is not True')

            return

        published = self.publish_safely(
            resources={
                c: p if os.path.isabs(p) else os.path.join(self.CHECKOUT_PATH, p) for (c, p) in resources.items()},
            comment='For branch {}'.format(self.Parameters.branch),
            copy=True)

        for resource in published.values():
            for attr_name, attr_value in resources_attrs.items():
                setattr(resource, attr_name, attr_value)

        return published
