import logging
import os
import shutil
import subprocess
from typing import Dict, Type

from .config import read_config
from .mk_writer import AndroidMkFileWriter, ProductMkFileWriter
from .models import DownloadableAppModule, ArtifactFolderInfo, \
    DownloadInfo, WebDownloadInfo, TeamcityDownloadInfo, SandboxResourceInfo
from .engine import DownloadEngine
from .http_engine import HTTPDownloadEngine
from .teamcity_engine import TeamcityDownloadEngine
from .sandbox_engine import SandboxDownloadEngine

logger = logging.getLogger(__name__)


def read_platform_generation(work_dir):
    searched_property = 'ro.yndx.build.generation='
    generation = None
    with open(os.path.join(work_dir, 'yandex.mk'), 'r') as read_obj:
        for line in read_obj:
            if searched_property in line:
                ind = line.index(searched_property)
                generation = line[ind + len(searched_property):len(line) - 1]

    return generation


def read_apk_generation(path):
    # requires container with pyaxmlparser installed
    commands = [
        "from pyaxmlparser import APK",
        "print(APK('{}').get_attribute_value('meta-data', 'value', name='yndx.build.generation'))".format(path),
    ]
    output = subprocess.check_output(['python3', '-c', ';'.join(commands)]).strip()
    if output == str(None):
        return None
    else:
        return output


def download_prebuilt_artifacts(task, work_dir, out_dir, config_path, config_parameters, force_vendor_modules=False):
    yaml_config = read_config(task, config_path, config_parameters)
    out_dir = os.path.join(work_dir, out_dir)

    if not os.path.exists(out_dir):
        os.makedirs(out_dir)

    android_mk = AndroidMkFileWriter(yaml_config.android_mk_file, out_dir)
    product_mk = ProductMkFileWriter(yaml_config.product_mk_file, out_dir)
    platform_generation = read_platform_generation(work_dir)

    download_engines = {
        WebDownloadInfo: HTTPDownloadEngine(),
        TeamcityDownloadInfo: TeamcityDownloadEngine(),
        SandboxResourceInfo: SandboxDownloadEngine(),
    }  # type: Dict[Type[DownloadInfo], DownloadEngine]

    for artifact in yaml_config.artifacts:
        download_engine = download_engines[artifact.default_artifact.__class__]

        if isinstance(artifact, DownloadableAppModule):
            if artifact.platforms is not None and task.config.platform not in artifact.platforms:
                continue

            app_metadata = artifact.metadata
            if os.path.exists(app_metadata.module_name):
                os.remove(app_metadata.module_name)
                logger.debug('removed old file')

            logger.debug('resolving {}'.format(app_metadata.module_name))

            download_info = artifact.default_artifact
            download_info.is_folder = False
            download_info.ext = 'apk'
            out_file_name = os.path.join(out_dir, app_metadata.module_name + '.apk')
            url, file_metadata = download_engine.download_artifact(download_info, out_file_name)[0]

            artifact_generation = read_apk_generation(out_file_name)
            if artifact_generation and artifact_generation != platform_generation:
                raise RuntimeError(
                    'Incomparable versions of generations: Platform={} | {}={}'.format(
                        platform_generation, app_metadata.module_name, artifact_generation))

            android_mk.write_mk_entry(
                app_metadata, '# Downloaded from {}\n# Meta: {}'.format(url, file_metadata), force_vendor_modules)
            product_mk.write_mk_entry(app_metadata)
            android_mk.flush()
            product_mk.flush()
            logger.debug('saved to {}'.format(out_file_name))
            logger.debug('-----')
        elif isinstance(artifact, ArtifactFolderInfo):
            out_folder_name = os.path.join(out_dir, artifact.folder_name)
            if os.path.exists(out_folder_name):
                shutil.rmtree(out_folder_name)
                logger.debug('removed old folder')

            logger.debug('resolving {}'.format(artifact.folder_name))
            download_info = artifact.default_artifact
            download_info.is_folder = True
            download_info.ext = artifact.ext

            download_engine.download_artifact(download_info, out_folder_name)
            logger.debug('saved to {}'.format(out_folder_name))
            logger.debug('-----')
