from typing import List, Dict, Any

import yaml
from sandbox import sdk2

from .models import DownloadableModule, DownloadableAppModule, \
    MkFileInfo, TeamcityDownloadInfo, WebDownloadInfo, SandboxResourceInfo, AppMetadata, ArtifactFolderInfo


class YamlConfig(object):
    def __init__(self, artifacts, android_mk_file, product_mk_file):
        # type: (List[DownloadableModule], MkFileInfo, MkFileInfo) -> None
        self.artifacts = artifacts
        self.android_mk_file = android_mk_file
        self.product_mk_file = product_mk_file


def get_class_constructor(obj_class):
    def _constructor(loader, node):
        values = loader.construct_mapping(node, deep=True)
        try:
            return obj_class(**values)
        except TypeError as ex:
            raise RuntimeError('Error constructing class <{}> from <{}>: {}'.format(obj_class.__name__, values, ex))

    return _constructor


def get_secret_constructor(task):
    def _constructor(loader, node):
        values = loader.construct_mapping(node)
        secret_path = task.yav_export_file(values["secret"], values["key"])
        with open(secret_path, 'r') as secret:
            return secret.read().strip()

    return _constructor


def get_parameter_constructor(parameter_values):
    def _constructor(loader, node):
        values = loader.construct_mapping(node)
        if 'default_value' in values:
            return parameter_values.get(values['key'], values['default_value'])
        else:
            return parameter_values[values['key']]

    return _constructor


def read_config(task, path, config_parameters=None):
    # type: (sdk2.Task, str, Dict[str, Any]) -> YamlConfig

    yaml.add_constructor('!parameter', get_parameter_constructor(config_parameters or {}))
    yaml.add_constructor('!yav', get_secret_constructor(task))
    yaml.add_constructor('!yaml_config', get_class_constructor(YamlConfig))
    yaml.add_constructor('!mk_file', get_class_constructor(MkFileInfo))
    yaml.add_constructor('!app_module', get_class_constructor(DownloadableAppModule))
    yaml.add_constructor('!app_meta', get_class_constructor(AppMetadata))
    yaml.add_constructor('!sandbox_resource', get_class_constructor(SandboxResourceInfo))
    yaml.add_constructor('!web_download', get_class_constructor(WebDownloadInfo))
    yaml.add_constructor('!teamcity_download', get_class_constructor(TeamcityDownloadInfo))
    yaml.add_constructor('!artifact_folder', get_class_constructor(ArtifactFolderInfo))

    with open(path) as f:
        return yaml.load(f, Loader=yaml.Loader)
