import logging
from typing import List, Tuple, Dict
from urllib3 import Retry

import requests
from requests.adapters import HTTPAdapter

from .engine import DownloadEngine
from .models import WebDownloadInfo


logger = logging.getLogger(__name__)


class HTTPDownloadEngine(DownloadEngine):
    def make_auth_headers(self, download_info):
        if download_info.auth_token:
            return {"Authorization": "OAuth {}".format(download_info.auth_token)}
        else:
            return {}

    def make_request(self, url, headers):
        # type: (str, Dict[str, str]) -> requests.Response
        response = requests_retry_session().get(url, headers=headers, allow_redirects=True)
        response.raise_for_status()
        return response

    def download_file(self, url, headers, dest):
        # type: (str, Dict[str, str], str) -> None
        response = self.make_request(url, headers)
        with open(dest, 'wb') as f:
            for chunk in response.iter_content(chunk_size=1024):
                if chunk:  # filter out keep-alive new chunks
                    f.write(chunk)

    def download_artifact(self, download_info, dest):
        # type: (WebDownloadInfo, str) -> List[Tuple[str, str]]
        logger.debug('downloading {}'.format(download_info.url))
        self.download_file(download_info.url, self.make_auth_headers(download_info), dest)
        return [(download_info.url, "")]


def requests_retry_session(
    retries=3,
    backoff_factor=0.3,
    status_forcelist=(100, 104, 500, 502, 504),
):
    session = requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session
