import re
import os
from datetime import datetime
from string import Template
from typing import List

from .models import MkFileInfo, AppMetadata


class MkFileWriter(object):
    def __init__(self, info, dest):
        # type: (MkFileInfo, str) -> None
        path = os.path.join(dest, info.name)
        self._file = open(path, "w+")
        self._substitutes = {
            'date': str(datetime.now().year)
        }
        if info.header:
            self._file.write(Template(info.header).safe_substitute(self._substitutes))

    def write(self, s):
        # type: (str) -> None
        self._file.write(s)

    def writelines(self, lines):
        # type: (List[str]) -> None
        self._file.writelines(lines)

    def flush(self):
        self._file.flush()

    def close(self):
        self._file.close()


def camel_case_split(s):
    # type: (str) -> str
    return ' '.join(re.findall(r'[A-Z](?:[a-z]+|[A-Z]*(?=[A-Z]|$))', s))


class AndroidMkFileWriter(MkFileWriter):
    def __init__(self, info, dest):
        # type: (MkFileInfo, str) -> None
        super(AndroidMkFileWriter, self).__init__(info, dest)
        self.write('LOCAL_PATH := $(call my-dir)\n\n')
        self.write('# Downloadable prebuilt apps\n\n')

    def close(self):
        self._file.write('\ninclude $(CLEAR_VARS)')
        super(AndroidMkFileWriter, self).close()

    def write_mk_entry(self, app_meta, extra_comment=None, force_vendor=False):
        # type: (AppMetadata, str, bool) -> None
        is_vendor_module = (force_vendor and not app_meta.system_only_module) or app_meta.vendor_module
        lines = [
            '# {}'.format(camel_case_split(app_meta.module_name)),
        ]
        if extra_comment:
            lines.append(extra_comment)
        lines.extend([
            'include $(CLEAR_VARS)',
            'LOCAL_MODULE := {}'.format(app_meta.module_name),
            'LOCAL_MODULE_OWNER := {}'.format(app_meta.owner),
            'LOCAL_SRC_FILES := $(LOCAL_MODULE).apk',
            'LOCAL_MODULE_TAGS := optional',
        ])
        if app_meta.certificate:
            lines.append('LOCAL_CERTIFICATE := {}'.format(app_meta.certificate))
        if is_vendor_module:
            lines.append('LOCAL_VENDOR_MODULE := {}'.format('true' if is_vendor_module else 'false'))
        if app_meta.overrides_packages:
            lines.append('LOCAL_OVERRIDES_PACKAGES := {}'.format(app_meta.overrides_packages))
        lines.append('LOCAL_MODULE_CLASS := {}'.format(app_meta.module_class))
        if app_meta.privileged:
            lines.append('LOCAL_PRIVILEGED_MODULE := {}'.format('true' if app_meta.privileged else 'false'))
        if app_meta.private_apis:
            lines.append('LOCAL_PRIVATE_PLATFORM_APIS := {}'.format('true' if app_meta.private_apis else 'false'))
        lines.extend([
            'LOCAL_MODULE_SUFFIX := $(COMMON_ANDROID_PACKAGE_SUFFIX)',
            'include $(BUILD_PREBUILT)',
            ''
        ])
        self.writelines([line + '\n' for line in lines])


class ProductMkFileWriter(MkFileWriter):
    def __init__(self, info, dest):
        # type: (MkFileInfo, str) -> None
        super(ProductMkFileWriter, self).__init__(info, dest)
        self.write('PRODUCT_PACKAGES += \\\n')

    def write_mk_entry(self, app_meta):
        # type: (AppMetadata) -> None
        self.write('    {} \\\n'.format(app_meta.module_name))
