import logging
import os
import re
from typing import Optional, List, Tuple
from urlparse import urlparse
from xml.etree import ElementTree

from .models import TeamcityDownloadInfo
from .http_engine import HTTPDownloadEngine

logger = logging.getLogger('teamcity-downloader')


class TeamcityArtifactBase(object):
    def __init__(self, base_url, project_id, build_id):
        # type: (str, str, str) -> None
        self.base_url = base_url
        self.project_id = project_id
        self.build_id = build_id


class ResolvedArtifact(object):
    def url(self):
        # type: () -> str
        raise NotImplementedError()


class TeamcityResolvedArtifact(ResolvedArtifact, TeamcityArtifactBase):
    def __init__(self, base_url, project_id, build_id, artifact_path):
        # type: (str, str, str, str) -> None
        TeamcityArtifactBase.__init__(self, base_url, project_id, build_id)
        ResolvedArtifact.__init__(self)
        self.artifact_path = artifact_path

    @staticmethod
    def from_download_info(info, artifact_path):
        # type: (TeamcityDownloadInfo, str) -> TeamcityResolvedArtifact
        return TeamcityResolvedArtifact(
            base_url=info.base_url,
            project_id=info.project_id,
            build_id=info.build_id,
            artifact_path=artifact_path
        )

    def __str__(self):
        return self.url()

    def url(self):
        formatted_id = self.build_id
        if self.build_id and self.build_id.isdigit():
            formatted_id = '{}:id'.format(self.build_id)
        return self.base_url + '/repository/download/' + self.project_id + '/' + formatted_id + '/' + self.artifact_path

    def build_upon(self, base_url=None, project_id=None, build_id=None,
                   artifact_path=None):
        # type: (str, str, str, str) -> TeamcityResolvedArtifact
        return TeamcityResolvedArtifact(
            base_url=base_url if base_url else self.base_url,
            project_id=project_id if project_id else self.project_id,
            build_id=build_id if build_id else self.build_id,
            artifact_path=artifact_path if artifact_path else self.artifact_path
        )


class TeamcityDownloadEngine(HTTPDownloadEngine):
    def resolve_last_successful_build_id(self, download_info):
        # type: (TeamcityDownloadInfo) -> Optional[TeamcityDownloadInfo]
        if download_info.build_id and download_info.build_id.isdigit():
            return download_info
        parsed_uri = urlparse(download_info.base_url)
        base_url = '{uri.scheme}://{uri.netloc}'.format(uri=parsed_uri)
        url = '{}/app/rest/builds/?locator='.format(base_url)
        if download_info.branch:
            url += 'branch:%s,' % download_info.branch
        url += 'buildType:%s,' % download_info.project_id
        if download_info.build_id == '.lastPinned':
            url += 'pinned:true,'
        url += 'status:success,' \
               'count:1' \
               '&fields=count,build(id,number,status)'
        logger.debug('locator url = {}'.format(url))
        response = self.make_request(url, self.make_auth_headers(download_info))
        tree = ElementTree.fromstring(response.content)
        build = tree.find('build')
        b_id = build.get('id')
        logger.debug('matched build id = {}'.format(b_id))
        return download_info.build_upon(build_id=b_id)

    def resolve_artifact(self, download_info):
        # type: (TeamcityDownloadInfo) -> List[TeamcityResolvedArtifact]
        logger.info("resolved branch: {}".format(download_info.branch))

        with_id = self.resolve_last_successful_build_id(download_info)
        if not with_id:
            raise RuntimeError('No last successful build found for {}'.format(download_info))

        ivy_artifact = TeamcityResolvedArtifact.from_download_info(with_id, 'teamcity-ivy.xml')
        logger.debug('retrieving ivy.xml {}'.format(ivy_artifact))
        response = self.make_request(ivy_artifact.url(), self.make_auth_headers(download_info))
        tree = ElementTree.fromstring(response.content)

        if not with_id:
            raise RuntimeError('failed to parse ivy.xml')

        pubs = tree.find('publications')
        if not pubs:
            raise RuntimeError('No publications found in ivy.xml for {}'.format(download_info))

        candidates = []
        result = []

        for publication in pubs.findall('artifact'):
            ext = publication.get('ext')
            if download_info.ext and download_info.ext == ext:
                art_path = publication.get('name') + '.' + ext
                candidates.append(art_path)
                compiled_pattern = re.compile(download_info.artifact_pattern)
                if compiled_pattern.search(art_path):
                    logger.debug('name = {} successfully matches {}'.format(art_path, download_info.artifact_pattern))
                    result.append(ivy_artifact.build_upon(artifact_path=art_path))

        if result:
            return result
        else:
            raise RuntimeError(
                'No artifacts in ivy.xml match {}, candidates = {}'.format(download_info.artifact_pattern, candidates))

    def get_artifact_metadata(self, resolved_artifact, download_info):
        # type: (TeamcityResolvedArtifact, TeamcityDownloadInfo) -> str
        parsed_uri = urlparse(resolved_artifact.base_url)
        base_url = '{uri.scheme}://{uri.netloc}'.format(uri=parsed_uri)
        url = '{}/app/rest/builds/{}'.format(base_url, resolved_artifact.build_id)

        logger.debug('metadata url = {}'.format(url))

        try:
            response = self.make_request(url, self.make_auth_headers(download_info))
            tree = ElementTree.fromstring(response.content)
            revisions = tree.find('revisions')
            revision = revisions.find('revision')
            commit_hash = revision.get('version')
            branch = revision.get('vcsBranchName')
            metadata = "built from {}, commit hash: {}".format(branch, commit_hash)
            logger.debug("meta: {}".format(metadata))

            return metadata
        except BaseException as ex:
            logger.error("metadata retrieval failed with an error, skipping: {}".format(ex))
            return ''

    def download_artifact(self, download_info, dest):
        # type: (TeamcityDownloadInfo, str) -> List[Tuple[str, str]]
        result = []
        resolved_list = self.resolve_artifact(download_info)
        if not resolved_list:
            raise RuntimeError('Failed to resolve {}'.format(download_info))

        if not download_info.is_folder and len(resolved_list) > 1:
            artifacts_str = list(map(lambda x: x.artifact_path, resolved_list))
            raise RuntimeError('is_folder = False, but more than 1 artifact found for pattern {}: {}'
                               .format(download_info.artifact_pattern, artifacts_str))

        for resolved in resolved_list:
            url = resolved.url()
            logger.debug('downloading {}'.format(url))

            download_to = dest
            if download_info.is_folder:
                # For example:
                # dest = 'downloadable/breakpad_symbols'
                # download_info.artifact_pattern = 'breakpad_symbols'
                # resolved.artifact_path = 'outputs/breakpad_symbols/foo/bar.so.sym'
                #
                # Then download_to should be 'downloadable/breakpad_symbols/foo/bar.so.sym'
                download_to = os.path.join(
                    dest, resolved.artifact_path.split(download_info.artifact_pattern)[1].lstrip('/'))

                os.makedirs(os.path.dirname(download_to))
            self.download_file(url, self.make_auth_headers(download_info), download_to)
            metadata = self.get_artifact_metadata(resolved, download_info)
            result.append((url, metadata))
        return result
