import json
import logging
import os
import shutil
import urlparse

from glob import glob

import sandbox.common.errors as sandbox_errors
import sandbox.projects.common.arcadia.sdk as arc_sdk
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.vcs.arc as vcs_arc
import sandbox.sandboxsdk.process as sandbox_process
import sandbox.sandboxsdk.ssh as sandbox_ssh
import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.build_types as build_types
import sandbox.projects.quasar.image_builds.artifact_downloader as artifact_downloader
import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.resource_types as qrt
import sandbox.projects.quasar.utils as quasar_utils
import sandbox.projects.quasar.utils.vcs as vcs_utils


class BaseImageBuildTask(
        sdk2.Task,
        binary_task.LastRefreshableBinary,
        quasar_utils.SafePublishingMixing,
        vcs_utils.RepoCheckoutMixin,
):
    class Parameters(sdk2.Parameters):
        kill_timeout = 60 * 60
        fail_on_any_error = True
        _lbrp = binary_task.binary_release_parameters(none=True)
        repository_url = sdk2.parameters.String('Repository url')
        repository_tag = sdk2.parameters.String('Repository tag')
        repository_user = sdk2.parameters.String('Repository user')
        repository_config_dir = sdk2.parameters.String('Arcadia url for repository config')
        repository_checkout_threads = sdk2.parameters.Integer('Number of threads used in repo sync')
        keep_resources = sdk2.parameters.Bool('Publish resources with ttl="inf"', default=False)
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Url for arcadia', default=None, required=False)
        with sdk2.parameters.Output:
            ota_images = sdk2.parameters.List("OTA images resources", value_type=sdk2.parameters.Integer, default=[])

    class BuildConfig(object):
        has_vcs_selector = False
        repository_user = None
        default_repository_tag = 'master'
        release_build_type = build_types.ImageBuildtype.USER
        default_repository_checkout_threads = 8

        def __init__(self, task):
            """
            :param BaseQuasarImageBuildTask task:
            """
            self.task = task

        @property
        def platform(self):
            raise NotImplementedError

        @property
        def checkout_path(self):
            return self.task.path('quasar')

        @property
        def ssh_private_key_vault_name(self):
            raise NotImplementedError

        @property
        def ssh_private_key_vault_owner(self):
            raise NotImplementedError

        @property
        def default_vcs(self):
            """ :rtype: str """
            raise NotImplementedError

        @property
        def default_repository_url(self):
            """ :rtype: str """
            raise NotImplementedError

        @property
        def default_buildroot_cache_platform(self):
            return 'common'

    def _get_default_repository_user(self):
        hostname = urlparse.urlsplit(self.repository_url).hostname
        if hostname.startswith('bb'):
            return 'git'
        elif hostname.startswith('gerrit'):
            return 'robot-quasar'
        else:
            return None

    @staticmethod
    def get_path_and_revision(arcadia_url):
        if "@" not in arcadia_url:
            arcadia_path = arcadia_url
            revision = sdk2.svn.Arcadia.info(arcadia_url)["commit_revision"]
        else:
            arcadia_path, revision = arcadia_url.split("@")
        return arcadia_path, revision

    def _setup_vcs(self):
        self.vcs = self.config.default_vcs
        if self.config.has_vcs_selector and self.Parameters.vcs:
            self.vcs = self.Parameters.vcs
        self.repository_url = self.config.default_repository_url
        self.repository_tag = self.config.default_repository_tag
        self.repository_checkout_threads = self.config.default_repository_checkout_threads

        self.checkout_path = str(self.config.checkout_path)
        self.ssh_private_key_vault_name = self.config.ssh_private_key_vault_name
        self.ssh_private_key_vault_owner = self.config.ssh_private_key_vault_owner

        self.repo_custom_manifest_path = None
        manifest_origin = None
        if self.Parameters.repository_config_dir:
            local_repository_config_dir_path = str(self.path("repository_config_dir"))
            arcadia_dir_string = "/arcadia/"
            config_dir_relative_path = self.Parameters.repository_config_dir
            arcadia_index = config_dir_relative_path.find(arcadia_dir_string)
            if arcadia_index != -1:
                config_dir_relative_path = config_dir_relative_path[arcadia_index + len(arcadia_dir_string):]
            if "@" in config_dir_relative_path:
                config_dir_relative_path = config_dir_relative_path.split("@")[0]
            try:
                with arc_sdk.mount_arc_path(
                    self.Parameters.checkout_arcadia_from_url, use_arc_instead_of_aapi=True
                ) as mount_path:
                    shutil.copytree(
                        os.path.join(mount_path, config_dir_relative_path),
                        local_repository_config_dir_path
                    )
            except vcs_arc.ArcCommandFailed:
                arcadia_path, revision = self.get_path_and_revision(self.Parameters.checkout_arcadia_from_url)
                if arcadia_dir_string not in self.Parameters.repository_config_dir:
                    arcadia_path += self.Parameters.repository_config_dir
                else:
                    arcadia_path = self.Parameters.repository_config_dir.split("@")[0]
                sdk2.svn.Arcadia.export(
                    "@".join((arcadia_path, revision)),
                    local_repository_config_dir_path
                )

            with open(os.path.join(local_repository_config_dir_path, "repository.json")) as repository_json:
                repository_config = json.load(repository_json)

                self.repository_url = repository_config["url"]
                self.repository_tag = repository_config["ref"]

                if self.vcs == vcs_utils.VCS.REPO and self.config.platform in quasar_platform.FIRMWARE_REPO_TRACKING:
                    self.repo_custom_manifest_path = os.path.join(
                        local_repository_config_dir_path,
                        repository_config["ref"],
                        "manifest.xml"
                    )
                    manifest_origin = "{} from arcadia url {}".format(
                        os.path.join(config_dir_relative_path, repository_config["ref"], "manifest.xml"),
                        self.Parameters.checkout_arcadia_from_url,
                    )

        logging.info(self.config.repository_user)
        logging.info(self.config.ssh_private_key_vault_name)
        self.repository_user = self.config.repository_user or self._get_default_repository_user()

        if self.vcs == vcs_utils.VCS.REPO and self.Parameters.custom_manifest:
            arcadia_path, revision = self.get_path_and_revision(self.Parameters.custom_manifest)
            custom_manifest_local_path = os.path.abspath(os.path.basename(arcadia_path))
            export_url = "@".join((arcadia_path, revision))
            try:
                sdk2.svn.Arcadia.export(
                    export_url,
                    custom_manifest_local_path
                )
            except sdk2.svn.SvnPathNotExists:
                raise sandbox_errors.TaskFailure(
                    "Custom manifest file does not exist.\n{}".format(export_url))
            self.repo_custom_manifest_path = custom_manifest_local_path
            manifest_origin = export_url

        if self.Parameters.repository_url:
            self.repository_url = self.Parameters.repository_url
        if self.Parameters.repository_tag:
            self.repository_tag = self.Parameters.repository_tag
        if self.Parameters.repository_user:
            self.repository_user = self.Parameters.repository_user
        if self.vcs == vcs_utils.VCS.REPO and self.Parameters.repository_checkout_threads:
            self.repository_checkout_threads = self.Parameters.repository_checkout_threads

        logging.info(
            "Set up checkout params: vcs=%s, url=%s, tag=%s, user=%s",
            self.vcs, self.repository_url, self.repository_tag, self.repository_user,
        )
        if self.repo_custom_manifest_path:
            logging.info("Repo manifest will be overriden with %s", manifest_origin)

    def _read_version_file(self, path):
        """
        :param str path: path to version file
        :returns: tuple with version parts
        """
        parts = self.path(path).read_text().strip().split('.')

        if len(parts) != 2:
            raise ValueError('File %s has invalid version (not two dot-separated parts) of %s' % (path, parts))

        return parts

    def _publish(self, resources=None, resources_attrs=None, copy_resources=True):
        """
        A publish phase after the build

        :param dict resources: a {<resource_class> : <resource_path>}
            map for resources to be published after phase builds
            <resource_path> should be relative to the self.CHECKOUT_PATH or absolute path
            <resource_path> is glob-expanded via `glob.glob`
        :param dict resources_attrs: a {name: value} dict with extra attributes to be added to all resources
        :param bool copy_resources: if to copy resources instead of moving

        :return: map of published resources {<resource_class>: <resource_object>}
        """
        resources = resources or {}
        resources_attrs = resources_attrs or {}

        def try_expand(a_path):
            """
            Expands given `a_path` to a first really exising file.
            If no file is found `a_path` is returned as-is.
            """
            expanded = glob(a_path)

            if expanded:
                return expanded[0]
            else:
                logging.warning('No files match %s', a_path)
                # show unexpanded path so publishing fails
                return a_path

        if self.config.platform in quasar_platform.FIRMWARE_REPO_TRACKING and self.repo_custom_manifest_path:
            comment = 'For custom manifest (branch {})'.format(self.repository_tag)
        else:
            comment = 'For branch {}'.format(self.repository_tag)

        published = self.publish_safely(
            resources={
                c: try_expand(p if os.path.isabs(p) else os.path.join(self.checkout_path, p))
                for (c, p) in resources.items()
            },
            comment=comment,
            copy=copy_resources,
        )

        for resource in published.values():
            for attr, attr_value in resources_attrs.items():
                if hasattr(resource, attr):
                    setattr(resource, attr, attr_value)

            if self.Parameters.keep_resources:
                resource.ttl = "inf"

        return published

    def on_execute(self):
        # Not using sandbox hooks because this doesn't work as expected in priveleged tasks
        # https://wiki.yandex-team.ru/sandbox/cookbook/#privileged-task
        # https://wiki.yandex-team.ru/sandbox/tasks/#execution-stages

        self._on_prepare()
        self._on_execute()
        self._on_success()

    def _on_success(self):
        self.Parameters.ota_images = self._ota_images

    @property
    def config(self):
        # Only reason of this property is the ability to override it with another type hint
        return self._config

    def _on_prepare(self):
        self._config = self.BuildConfig(self)
        self._setup_vcs()
        self._ota_images = []

    def _checkout(self):
        if self.vcs == vcs_utils.VCS.GIT:
            self.git_checkout()
        elif self.vcs == vcs_utils.VCS.REPO:
            self.repo_checkout()
        else:
            raise sandbox_errors.TaskFailure("VCS '{}' is not supported".format(self.vcs))

    def _build(self, build_arguments=None, environment=None):
        """
        A build phase matching one from the `build_all_ci.sh` script
        :param build_arguments: extra arguments to pass to build script
        :param dict environment: extra env to be passed to the `build_all_ci.sh` script

        :return: map of published resources {<resource_class>: <resource_object>}
        """
        build_arguments = build_arguments or []
        environment = environment or {}

        logging.info('Running the %s phase...', ''.join(build_arguments))

        env = os.environ.copy()
        env.update(environment)

        log_prefix = build_arguments[0] if build_arguments else 'build'
        self._run_build_script(build_arguments, env, log_prefix)

    def _publish_repo_manifest(self):
        if self.repo_custom_manifest_path:
            manifest_path = self.repo_custom_manifest_path
        else:
            manifest_path = str(self.config.checkout_path / 'manifest.xml')
            get_manifest_command = 'repo manifest -r --suppress-upstream-revision -o {}'.format(manifest_path)
            sandbox_process.run_process(
                ['bash', '-c', get_manifest_command],
                log_prefix=self.config.platform,
                work_dir=self.checkout_path,
            )

        self._publish(resources={qrt.QuasarRepoManifest: manifest_path})


class BaseAndroidImageBuildTask(BaseImageBuildTask):
    class Parameters(BaseImageBuildTask.Parameters):
        with sdk2.parameters.Group("Artifact downloader") as tv_apps_block:
            artifact_downloader_arcadia_config = sdk2.parameters.String("Artifacts config file in arcadia")
            artifact_downloader_dependencies = sdk2.parameters.Dict("Artifact dependencies")
            artifact_downloader_dependencies_json = sdk2.parameters.JSON("Artifact dependencies as json dict")
            artifact_downloader_force_vendor_modules = sdk2.parameters.Bool(
                "Put applications to vendor partition where applicable")

            kinopoisk_branch = sdk2.parameters.String("Branch for kinopoisk app", default="")

    def _download_prebuilt(self):
        with arc_sdk.mount_arc_path(
            self.Parameters.checkout_arcadia_from_url, use_arc_instead_of_aapi=True
        ) as mount_path:
            with sandbox_ssh.Key(self, self.ssh_private_key_vault_owner, self.ssh_private_key_vault_name):
                config_path = os.path.join(mount_path, self.Parameters.artifact_downloader_arcadia_config)
                if self.Parameters.artifact_downloader_dependencies_json:
                    config_parameters = self.Parameters.artifact_downloader_dependencies_json
                else:
                    config_parameters = self.Parameters.artifact_downloader_dependencies
                if self.Parameters.kinopoisk_branch:
                    config_parameters['kinopoisk_branch'] = self.Parameters.kinopoisk_branch

                work_dir = os.path.join(self.checkout_path, self.config.yandex_prebuilt_repo_path)
                download_dirname = 'downloadable'
                artifact_downloader.download_prebuilt_artifacts(
                    self,
                    work_dir=work_dir,
                    out_dir=download_dirname,
                    config_path=config_path,
                    config_parameters=config_parameters,
                    force_vendor_modules=self.Parameters.artifact_downloader_force_vendor_modules,
                )
                resources = {
                    self.config.downloaded_artifacts_config: os.path.join(
                        self.checkout_path, config_path),
                    self.config.downloaded_artifacts: os.path.join(
                        self.checkout_path, os.path.join(work_dir, download_dirname)),
                }
                self._publish(resources=resources)
