import logging
import os

import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.process as sandbox_process
import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.build_types as build_types
import sandbox.projects.quasar.image_builds.base_image_build as base
import sandbox.projects.quasar.resource_types as qrt
import sandbox.projects.quasar.resource_types.tv as tv_resource_types
import sandbox.projects.quasar.utils as quasar_utils
import sandbox.projects.quasar.utils.vcs as quasar_vcs


class QuasarBuildAmlogicAndroidImage(
        quasar_utils.YAVExportMixin,
        base.BaseAndroidImageBuildTask
):
    class BuildConfig(base.BaseAndroidImageBuildTask.BuildConfig):
        platform = None
        ssh_private_key_vault_name = 'robot-quasar-ssh-private-key'
        ssh_private_key_vault_owner = 'QUASAR'
        place_default_tag_in_version = False
        has_soc_selector = False
        default_vcs = quasar_vcs.VCS.REPO
        default_repository_url = 'ssh://gerrit.yandex-team.ru/unity/platform/manifest'
        daemons_dest_path = None
        strip_tool_path = None

        @property
        def yandex_prebuilt_repo_path(self):
            return os.path.join(str(self.checkout_path), 'vendor/yandex/prebuilt')

    class Parameters(base.BaseAndroidImageBuildTask.Parameters):
        kill_timeout = 3 * 60 * 60  # 3h

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool(
                'Suspend task for debug purposes just before actual build starts', default=False)

        with sdk2.parameters.String('Build type', required=True, default=build_types.ImageBuildtype.USER) as build_type:
            for image_build_type in list(build_types.ImageBuildtype):
                build_type.values[image_build_type] = image_build_type

        apps_major_version = sdk2.parameters.Integer('Apps major version number. Usually branch number.')
        apps_minor_version = sdk2.parameters.Integer('Apps minor version number. Usually tag number in branch.')

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")

        sign = sdk2.parameters.Bool('Build signed image and OTA', default=True)
        factory = sdk2.parameters.Bool('Build factory image', default=False)
        hw_test_mode = sdk2.parameters.Bool('Build a hw test firmware, without quasar', default=False)
        secureboot = sdk2.parameters.Bool('Build secureboot image and OTA', default=True)

        teamcity_token_key_secret = sdk2.parameters.String(
            'YAV secret with teamcity token: robot-quasar-teamcity-token',
            default='ver-01f6237f9s51deh8nt2b5kpt21')

        target_files = sdk2.parameters.Bool('Save target_files to resource', default=False)

    class Requirements(
            quasar_utils.YAVExportMixin.Requirements,
            base.BaseAndroidImageBuildTask.Requirements
    ):
        client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.SSD  # repo is big
        disk_space = 200 * 1024  # 200 Gb

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def _on_execute(self):
        self._checkout()

        artifacts = (
            self.Parameters.checkout_arcadia_from_url and
            self.Parameters.artifact_downloader_arcadia_config and
            (
                self.Parameters.artifact_downloader_dependencies or
                self.Parameters.artifact_downloader_dependencies_json
            )
        )

        if artifacts:
            self._download_prebuilt()

        # abspath so _place / _publish wont try to prepend checkout dir
        ota_out, image_out, target_files_out = [
            os.path.abspath(str(self.path(p))) for p in
            ['ota.zip', 'aml.img', 'target_files.zip']
        ]

        build_version = str(self.id)

        build_args = [
            '--build_type', self.Parameters.build_type,
            '--build_version', build_version,
            '--image_dest', image_out,
            '--ota_dest', ota_out,
            '--verbose',
            '--clean',
        ]

        if self.config.has_soc_selector and self.Parameters.soc:
            build_args += ['--platform', self.Parameters.soc]

        if self.Parameters.target_files:
            build_args += ['--target_files_dest', target_files_out]

        if self.Parameters.sign:
            build_args += ['--sign', '--sign_keys', self.yav_export_file(self.Parameters.sign_keys_secret, 'keys.tgz')]
            build_args += ['--tdk_key', self.yav_export_file(str(self.Parameters.tdk_key_secret), 'tdk_keys.tgz')]

        if self.Parameters.build_type == 'user':
            build_args += ['--rabbit_hole_key', self.yav_export_file(
                str(self.Parameters.rabbit_hole_key_secret), 'rh_public.pem')]

        if self.Parameters.secureboot:
            build_args += ['--secureboot', '--secureboot_key', self.yav_export_file(
                str(self.Parameters.secureboot_keys_secret), 'aml-user-key.sig')]

        if bool(self.Parameters.avb_key_secret):
            avb_rsa_key = self.yav_export_file(str(self.Parameters.avb_key_secret), 'avb_private.pem')
            build_args += ['--avb-key', avb_rsa_key]

        if self.Parameters.factory:
            build_args += ['--factory']

        if self.Parameters.hw_test_mode:
            build_args += ['--hw_test_mode']

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Build args would be <%s>', ' '.join(build_args))
            self.suspend()

        self._build(build_args, self.get_build_script_env())

        resources = {
            self.ArtifactsConfig.image_resource_type: image_out,
            self.ArtifactsConfig.ota_resource_type: ota_out,
        }

        if self.Parameters.target_files:
            resources[self.ArtifactsConfig.target_files_resource] = target_files_out

        attrs = dict(
            buildtype=self.Parameters.build_type,
            signed=bool(self.Parameters.sign),
            factory=bool(self.Parameters.factory),
            secureboot=bool(self.Parameters.secureboot),
            hw_test_mode=bool(self.Parameters.hw_test_mode),
            repository_tag=self.repository_tag,
            version=build_version,
            verbose_version=self._determine_version(),
        )

        self._publish(
            resources=resources,
            resources_attrs=attrs
        )

        self._publish_repo_manifest()

    def _run_build_script(self, build_arguments, env, log_prefix):
        sandbox_process.run_process(
            [
                'bash',
                os.path.join(self.checkout_path, 'build_all_ci.sh'),
            ] + build_arguments,
            log_prefix=log_prefix,
            environment=env,
        )

    def get_build_script_env(self):
        env = os.environ.copy()
        env.update({
            'TMPDIR': env['TMP'],   # Fix env variables for clang. TODO: move to lxc container
            'BUILD_NUMBER': str(self.id),
            'BRANCH_NAME': self.Parameters.repository_tag or self.config.default_repository_tag,
            'LC_ALL': 'en_US.UTF-8',    # Workaround for UnicodeDecodeError
            'LC_CTYPE': 'en_US.UTF-8',  # Workaround for UnicodeDecodeError
        })
        return env

    def _determine_version(self):
        """
        :returns: a `str` version for this build.
        Version is defined as
        `r_1.q_1.r_2.q_2.task_id.build_date[.repository_tag].fw_build_type.apps_build_type[.factory_suffix]`,
        where:
            * r_1, r_2 - firmware repo version
            * q_1, q_2 - quasar repo version (q_1 is branch number, q_2 is tag number)
            * task_id - this task id,
            * build_date - date of the build in the form YYYYMMDD
            * repository_tag - optional repository tag (e.g. 18 for build from system branch yandexmodule_2/18)
            * fw_build_type - build type of firmware (USER/ENG/USERDEBUG)
            * apps_build_type - build type of Android apps (USER/PRESTABLE/USERDEBUG)
            * factory_suffix - optional suffix for factory firmware
        """
        f_version_path = self.config.checkout_path / 'VERSION'
        try:
            f_version = self._read_version_file(f_version_path)
        except Exception as err:  # noqa
            logging.warning('Cannot read system repo version from file %s. Error: %s', f_version_path, err)
            f_version = ("0", "0")

        if self.Parameters.apps_major_version or self.Parameters.apps_minor_version:
            q_version = (str(self.Parameters.apps_major_version), str(self.Parameters.apps_minor_version))
        else:
            q_version = ('1000', '1000')

        suffix = [str(self.id), self.created.strftime('%Y%m%d')]

        if self.repository_tag != self.config.default_repository_tag or self.config.place_default_tag_in_version:
            if self.repository_tag.startswith(self.config.platform):
                suffix.append(self.repository_tag[len(self.config.platform):].lstrip('/-_'))
            else:
                suffix.append(self.repository_tag)

        if self.Parameters.build_type != self.config.release_build_type:
            suffix.append(str(self.Parameters.build_type).upper())

        if self.Parameters.factory:
            suffix.append('FACTORY')

        version = '.'.join(['.'.join(pair) for pair in zip(f_version, q_version)] + suffix)
        logging.info('determined version to be %s', version)

        return version

    def _publish(self, resources=None, resources_attrs=None, copy_resources=True):
        published = super(QuasarBuildAmlogicAndroidImage, self)._publish(
            resources=resources,
            resources_attrs=resources_attrs,
            copy_resources=copy_resources,
        )

        for resource in published.values():
            if (
                    (isinstance(resource, tv_resource_types.AndroidTvImageBase) and resource.image_type == 'ota') or
                    isinstance(resource, qrt._QuasarOTABase)
            ):
                resource.crc32_checksum = quasar_utils.get_resource_crc32(resource)
                self._ota_images += [resource.id]

        return published
