import logging
import os
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import process as sandbox_process

from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_image_build
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar.utils import vcs as vcs_utils


class QuasarBaseBuildrootImageBuildTask(
        quasar_image_build.BaseQuasarImageBuildTask,
        quasar_utils.YAVExportMixin
):
    class Parameters(quasar_image_build.BaseQuasarImageBuildTask.Parameters):
        kill_timeout = 1 * 60 * 60  # 1h

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool(
                'Suspend task for debug purposes just before actual build starts',
                default=False)

        ota_with_uboot = sdk2.parameters.Bool('Build OTA with uboot', default=True)

        buildroot_defconfig = sdk2.parameters.String(
            'Buildroot defconfig name for main image',
            description='Without suffix "_defconfig" and you still need to specify build_type. '
                        'Don`t forget to specify buildroot_recovery_defconfig',
            default='')

        buildroot_recovery_defconfig = sdk2.parameters.String(
            'Buildroot defconfig name for recovery image',
            description='Without suffix "_defconfig" and you still need to specify build_type. '
                        'Don`t forget to specify buildroot_defconfig',
            default='')

    class Requirements(quasar_utils.YAVExportMixin.Requirements,
                       quasar_image_build.BaseQuasarImageBuildTask.Requirements):
        client_tags = ctc.Tag.SSD
        disk_space = 30 * 1024  # 30 Gb

    class BuildConfig(quasar_image_build.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def default_vcs(self):
            return vcs_utils.VCS.REPO

        @property
        def default_repository_url(self):
            return 'ssh://gerrit.yandex-team.ru/yandex-io/buildroot-manifest'

        @property
        def buildroot_cache_dest_path(self):
            return self.checkout_path / 'dist'

        @property
        def buldroot_cache_platform(self):
            return None

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    @property
    def buildroot_config_name(self):
        raise NotImplementedError

    @property
    def buildroot_recovery_config_name(self):
        raise NotImplementedError

    @property
    def with_quasar_daemons(self):
        raise NotImplementedError

    def prepare_build_command(self):
        raise NotImplementedError

    def publish_resources(self):
        raise NotImplementedError

    def yav_export_file(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of file to export
        :return str: path to exported file
        """
        return os.path.join(self.yav_export(secret_or_version, key), key)

    def yav_export_tar(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of tar archive to export
        :return str: path to exported dir
        """
        export_path = self.yav_export(secret_or_version, key)
        tar_path = os.path.join(export_path, key)
        extract_dir = tempfile.mkdtemp(prefix=key, dir=export_path)
        with tarfile.open(tar_path) as tar:
            tar.extractall(extract_dir)
        return extract_dir

    def publish_image(self, resource_type, secureboot, fastboot, version):
        if fastboot:
            image_file = 'fastboot-signed.img' if secureboot else 'fastboot.img'
        else:
            image_file = 'upgrade-signed.img' if secureboot else 'upgrade.img'

        image_path = os.path.join('out', self.buildroot_config_name, 'images', image_file)

        if version is None:
            version = self._read_version_file(self.config.checkout_path / 'VERSION')

        self._publish(
            resources={
                resource_type: image_path
            },
            resources_attrs=dict(
                buildtype=self.Parameters.build_type,
                version=version,
                signed=secureboot
            )
        )

    def publish_ota_image(self, resource_type, secureboot, version):
        if secureboot:
            image_file = 'yandex_io_ota.swu'
        else:
            image_file = 'yandex_io_ota-unsecured.swu'

        ota_image_path = os.path.join('out', self.buildroot_config_name, 'images', image_file)

        return self._publish(
            resources={
                resource_type: ota_image_path
            },
            resources_attrs=dict(
                version=version,
                signed=secureboot,
                buildtype=self.Parameters.build_type
            )
        )

    def _on_execute(self):
        self._checkout()

        self._place_package_cache()

        if self.with_quasar_daemons:
            self._place_daemons()
            self._place_daemons(factory=True)
            self._place_version(self._determine_version())

            factory_version = self._determine_version(factory=True)
            factory_version += '.factory'
            self._place_version(factory_version, path=self.config.factory_daemons_dest_path)

        build_command = self.prepare_build_command()

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            self.suspend()

        logging.info('Execute command %s', build_command)
        sandbox_process.run_process(
            ['bash', '-c', build_command],
            log_prefix=self.config.platform,
            work_dir=self.checkout_path,
        )

        self.publish_resources()

        self._publish_repo_manifest()
