import os

from sandbox import sdk2
from sandbox.projects import resource_types as srt
from sandbox.sandboxsdk import paths as sandbox_paths
from sandbox.sandboxsdk import process as sandbox_process

from sandbox.projects.quasar import platform as quasar_platform
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar.image_builds.base_image_build import linkplay as linkplay_image_build
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_image_build
from sandbox.projects.quasar.utils import vcs as vcs_utils


class QuasarBaseJblImageBuildTask(linkplay_image_build.QuasarLinkplayImageBuildTask):
    class Parameters(linkplay_image_build.QuasarLinkplayImageBuildTask.Parameters):
        param_creator = quasar_image_build.ParameterCreator(None)
        _container = quasar_utils.LastStableContainer(
            'Execution container', resource_type=qrt.QuasarJblLinkLxcImage)
        kill_timeout = 60 * 60

        keys_yav_version = None
        toolchain_resource = quasar_utils.LastResourceWithAttrs(
            'Toolchain to get strip tool',
            resource_type=srt.OTHER_RESOURCE,
            attrs=dict(
                toolchain_platform='jbl_link',
            )
        )
        custom_manifest = param_creator.create_custom_manifest_parameter()

    class Requirements(linkplay_image_build.QuasarLinkplayImageBuildTask.Requirements):
        privileged = True
        disk_space = 55 * 1024  # 55 Gb

    class BuildConfig(linkplay_image_build.QuasarLinkplayImageBuildTask.BuildConfig):
        @property
        def jbl_platform(self):
            if self.platform == quasar_platform.Platform.JBL_LINK_MUSIC:
                return 'music'
            elif self.platform == quasar_platform.Platform.JBL_LINK_PORTABLE:
                return 'portable'

        @property
        def image_resource_type(self):
            """ :rtype: quasar_resources._QuasarImageBase """
            raise NotImplementedError

        @property
        def base_dest_path(self):
            return self.checkout_path / 'device' / 'legacy' / 'A118_yandex_hm_link{}'.format(self.jbl_platform)

        @property
        def daemons_dest_path(self):
            return self.base_dest_path / 'linkplay' / 'system' / 'vendor' / 'quasar'

        @property
        def keys_dest_path(self):
            return self.base_dest_path / 'secure_boot'

        @property
        def default_vcs(self):
            return vcs_utils.VCS.REPO

        @property
        def default_repository_url(self):
            return 'ssh://gerrit.yandex-team.ru/yandex-io/linkplay/manifest'

        @property
        def default_repository_tag(self):
            return 'jbl-portable-link'

        @property
        def base_out_path(self):
            return self.checkout_path / 'out' / 'a118_yandex_hm_link{}'.format(self.jbl_platform) / 'image'

        @property
        def build_artifacts(self):
            base_path = self.base_out_path
            return {
                self.image_resource_type: str(base_path / 'a118_all_burning.zip'),
                self.ota_resource_type: str(base_path / 'SECURE_uImage_ota_moduleMulti'),
                qrt.QuasarJblReleaseNotes: str(base_path / 'release.txt'),
                qrt.QuasarJblDebugNotes: str(base_path / 'debug.txt'),
            }

        @property
        def build_params(self):
            return [
                '--daemons-revision', self.task.Parameters.quasar_daemons.svn_revision,
                '--build-type', self.task.Parameters.build_type.upper(),
                '--platform-type', self.jbl_platform,
                '--repo-path', '.',
                '--keys-path', str(self.keys_dest_path),
            ]

        @property
        def strip_tool_path(self):
            toolchain_path = self.task.path('toolchain')
            toolchain_path.mkdir(parents=True)

            toolchain_zip = str(sdk2.ResourceData(self.task.Parameters.toolchain_resource).path)
            sandbox_process.run_process(
                'tar -xvzf {} -C {}'.format(toolchain_zip, toolchain_path), log_prefix="decompress toolchain")

            return toolchain_path / 'usr' / 'bin' / 'arm-poky-linux-gnueabi' / 'arm-poky-linux-gnueabi-strip'

    @property
    def config(self):
        # type: () -> BuildConfig
        return self._config

    def _place_resources(self):
        super(QuasarBaseJblImageBuildTask, self)._place_resources()
        self._place_keys()

    def _place_keys(self):
        keys = dict()
        keys['sbc_key.pem'] = os.path.join(
            self.yav_export(self.Parameters.keys_yav_version, 'sbc_key.pem'), 'sbc_key.pem')
        keys['verified_key.crt'] = os.path.join(
            self.yav_export(self.Parameters.keys_yav_version, 'verified_key.crt'), 'verified_key.crt')
        keys['verified_key.pem'] = os.path.join(
            self.yav_export(self.Parameters.keys_yav_version, 'verified_key.pem'), 'verified_key.pem')

        for secret_name, _ in keys.items():
            dest_key_path = str(self.config.keys_dest_path / secret_name)
            sandbox_paths.remove_path(dest_key_path)
            os.symlink(keys[secret_name], dest_key_path)

    def _run_build_script(self, build_arguments, env, log_prefix):
        sandbox_process.run_process(
            [
                'bash',
                'build.sh',
            ] + build_arguments,
            log_prefix="build",
            work_dir=self.checkout_path,
            environment=env,
        )
