from sandbox import sdk2
from sandbox.common.types import client as ctc

from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_image_build
from sandbox.projects.quasar.utils import vcs as vcs_utils


class QuasarLinkplayImageBuildTask(quasar_image_build.BaseQuasarImageBuildTask, quasar_utils.YAVExportMixin):
    class Parameters(quasar_image_build.BaseQuasarImageBuildTask.Parameters):
        with sdk2.parameters.String(
            'Build type', required=True, default=build_types.LinkplayImageBuildType.RELEASE
        ) as build_type:
            for image_build_type in list(build_types.LinkplayImageBuildType):
                build_type.values[image_build_type] = image_build_type

    class Requirements(quasar_utils.YAVExportMixin.Requirements, sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.SSD  # repo is big
        disk_space = 2 * 1024  # 2 Gb

    class BuildConfig(quasar_image_build.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def default_vcs(self):
            return vcs_utils.VCS.GIT

        @property
        def daemons_dest_path(self):
            return self.checkout_path / 'target' / 'system' / 'vendor' / 'quasar'

        @property
        def release_build_type(self):
            return build_types.LinkplayImageBuildType.RELEASE

        @property
        def strip_tool_path(self):
            return self.checkout_path / 'arm-linux-gnueabihf-strip'

        @property
        def ota_resource_type(self):
            """ :rtype: resource_types._QuasarOTABase """
            raise NotImplementedError

        @property
        def build_artifacts(self):
            if self.task.Parameters.build_type == self.release_build_type:
                return {self.ota_resource_type: 'ota_without_adb.img'}
            else:
                return {self.ota_resource_type: 'ota_adb.img'}

        @property
        def build_params(self):
            if self.task.Parameters.build_type == self.release_build_type:
                phase = 'ota_without_adb'
            else:
                phase = 'ota_with_adb'

            return [phase, self.task.Parameters.build_type]

    @property
    def config(self):
        # type: () -> BuildConfig
        return self._config

    def _on_execute(self):
        self._checkout()
        self._place_resources()
        self._strip_binaries()

        quasar_sizes = quasar_utils.get_quasar_sizes(self.config.daemons_dest_path)

        version = self._determine_version()
        self._place_version(version)

        self._build(self.config.build_params)
        build_artifacts = self.config.build_artifacts

        resource_attrs = dict(
            buildtype=self.Parameters.build_type,
            version=version,
        )

        artifacts = self._publish(
            resources=build_artifacts,
            resources_attrs=resource_attrs,
        )

        if self.Parameters.push_sensors_to_solomon:
            quasar_utils.push_sizes_to_solomon(
                self.Parameters.quasar_daemons,
                quasar_sizes,
                artifacts[self.config.ota_resource_type],
            )

        if self.Parameters.ota_max_size:
            quasar_utils.check_ota_size(artifacts[self.config.ota_resource_type], self.Parameters.ota_max_size)
