import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common import errors as sandbox_errors
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.sandboxsdk import paths as sandbox_paths

from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import platform as quasar_platform
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar.image_builds import base_image_build
from sandbox.projects.quasar.utils import vcs as vcs_utils


class ParameterCreator(object):
    def __init__(self, platform):
        self.platform = platform

    def create_ota_max_size_parameter(self):
        return sdk2.parameters.Integer(
            'Max size for OTA, in megabytes', default=quasar_platform.OTA_SIZE_LIMITS[self.platform])

    def create_daemons_parameter(self, factory=False):
        attrs = dict(
            quasar_platform=self.platform,
            released=ctt.ReleaseStatus.TESTING,
        )
        resource_type = qrt.QuasarFactoryDaemons if factory else qrt.QuasarDaemons

        return quasar_utils.LastResourceWithAttrs(
            'Quasar {} binary daemons'.format('factory' if factory else ''),
            resource_type=resource_type,
            attrs=attrs
        )

    @staticmethod
    def create_vcs_parameter(supported_vcs):
        with sdk2.parameters.String('VCS') as vcs:
            for item in supported_vcs:
                vcs.values[item] = item
            return vcs

    @staticmethod
    def create_custom_manifest_parameter():
        return sdk2.parameters.String('Arcadia url for manifest')


class BaseQuasarImageBuildTask(
        base_image_build.BaseImageBuildTask,
        vcs_utils.GitCheckoutMixin,
):
    class Parameters(base_image_build.BaseImageBuildTask.Parameters):
        ota_max_size = None
        quasar_daemons = None
        quasar_daemons_factory = None

        push_sensors_to_solomon = sdk2.parameters.Bool('Push sensors to solomon', default=False)

        with sdk2.parameters.String('Build type', required=True, default=build_types.ImageBuildtype.USER) as build_type:
            for image_build_type in list(build_types.ImageBuildtype):
                build_type.values[image_build_type] = image_build_type

        with sdk2.parameters.Group('Internal') as internal_block:
            test = sdk2.parameters.Bool('Test some new and exciting feature', default=False)

            vcs = ParameterCreator.create_vcs_parameter([vcs_utils.VCS.GIT, vcs_utils.VCS.REPO])
            quasar_major_version = sdk2.parameters.Integer('Major release number for quasar daemons')
            quasar_minor_version = sdk2.parameters.Integer('Minor release number for quasar daemons')

        quasar_platform_revision = sdk2.parameters.String('Platform revision', default=None)

    class BuildConfig(base_image_build.BaseImageBuildTask.BuildConfig):
        has_vcs_selector = True
        ssh_private_key_vault_name = 'robot-quasar-ssh-private-key'
        ssh_private_key_vault_owner = 'QUASAR'
        place_default_tag_in_version = False
        daemons_source_path = None
        factory_daemons_source_path = None

        @property
        def daemons_dest_path(self):
            """ :rtype: sdk2.Path """
            raise NotImplementedError

        @property
        def factory_daemons_dest_path(self):
            """ :rtype: sdk2.Path """
            raise NotImplementedError

        @property
        def strip_tool_path(self):
            """ :rtype: sdk2.Path """
            raise NotImplementedError

        @property
        def buildroot_cache_dest_path(self):
            """ :rtype: sdk2.Path """
            raise NotImplementedError

        @property
        def buldroot_cache_platform(self):
            """ :rtype: sdk2.Path """
            raise NotImplementedError

        def load_package_config(self):
            if self.task.Parameters.quasar_daemons is None:
                return

            daemons_data = sdk2.ResourceData(self.task.Parameters.quasar_daemons)
            try:
                package_version = int(self.task.path(daemons_data.path, 'yandexio_package_version').read_text().strip())
                logging.info("Detected package version %d", package_version)
            except IOError:
                logging.info('yandexio_package_version not found in daemons package, assume package version 0')
                package_version = 0

            if package_version == 0:
                self.daemons_source_path = daemons_data.path
            elif package_version == 1:
                self.daemons_source_path = self.task.path(daemons_data.path, 'install_root')

        def load_factory_package_config(self):
            if self.task.Parameters.quasar_daemons_factory is not None:
                daemons_data = sdk2.ResourceData(self.task.Parameters.quasar_daemons_factory)
                self.factory_daemons_source_path = self.task.path(daemons_data.path, 'install_root')

    def _place_resources_to_location(self, resource_to_location, symlinks=False):
        """
        Base method to implement _place_resources:
        define resource-to-location map and call _place_resources_to_location

        TODO:
            Now this method is only used to place apk and has some specific apk-code
            This should be moved to something like BaseAndroidImageBuildTask
        """
        for resource, location in resource_to_location.items():
            if not resource:
                logging.warning('Skipping location <%s> as resource is not given', location)
                continue

            download_path = str(sdk2.ResourceData(resource).path)

            logging.info('Downloaded res %s to %s', resource, download_path)

            dest_path = os.path.join(self.checkout_path, location) if not os.path.isabs(location) else location
            sandbox_paths.remove_path(dest_path)
            # FIXME: that is an ugliest hack ever, and in an obscure place -- should assume proper resource stucture
            if (resource.type == qrt.QuasarApp or resource.type == qrt.QuasarStub) and os.path.isdir(download_path):
                apks = [f for f in os.listdir(download_path) if f.endswith('.apk')]
                # YA_PACKAGE returns directory. It should contain only one apk for now
                if len(apks) == 1:
                    download_path = os.path.join(download_path, apks[0])
                else:
                    raise ValueError('Resource %s should contain exatly one .apk file but found: %s' % (resource, apks))
            sandbox_paths.copy_path(download_path, dest_path, symlinks=symlinks)

    def _determine_version(self, f_version_file=None, factory=False):
        """
        :returns: a `str` version for this build.

        Version is defined as `r_1.q_1.r_2.q_2.task_id.build_date`, where:
            * firmware repo version is `f_1.f_2`
            * quasar repo version is `q_1.q_2`
            * `task_id` is id of current task
            * `build_date` is date in format `20180305`
        """
        f_version_path = f_version_file or self.config.checkout_path / 'VERSION'
        f_version = self._read_version_file(f_version_path)
        daemons_path = self.config.factory_daemons_dest_path if factory else self.config.daemons_dest_path
        q_version = self._read_version_file(daemons_path / 'VERSION')

        if self.Parameters.quasar_major_version and self.Parameters.quasar_minor_version:
            q_version = (str(self.Parameters.quasar_major_version), str(self.Parameters.quasar_minor_version))

        suffix = [str(self.id), self.created.strftime('%Y%m%d')]

        # TODO: when relying on exact commit use a different technique
        if self.repository_tag != self.config.default_repository_tag or self.config.place_default_tag_in_version:
            if self.repository_tag.startswith(self.config.platform):
                suffix.append(self.repository_tag[len(self.config.platform):].lstrip('/-_'))
            else:
                suffix.append(self.repository_tag)

        if self.Parameters.build_type != self.config.release_build_type:
            suffix.append(str(self.Parameters.build_type).upper())

        version = '.'.join(map('.'.join, zip(f_version, q_version)) + suffix)
        logging.info('determined version to be %s', version)

        return version

    def _place_version(self, version, path=None):
        """
        :param str version: a version to replace

        Replaces version placeholder in certain subtrees of sources
        """

        target_path = path or self.config.daemons_dest_path

        # TODO: Drop after moving to a version_full file
        sandbox_process.run_process(
            "find . -name 'quasar*cfg' | " +
            "xargs sed -i 's^__QUASAR_VERSION_PLACEHOLDER__^{}^g'".format(version),
            work_dir=str(target_path),
            shell=True,
            log_prefix='replace_version',
        )

        with open(os.path.join(str(target_path), "version_full"), "w+") as f:
            f.write(version)

    def _build(self, build_arguments=None, environment=None):
        """
        A build phase matching one from the `build_all_ci.sh` script
        :param build_arguments: extra arguments to pass to build script
        :param dict environment: extra env to be passed to the `build_all_ci.sh` script

        :return: map of published resources {<resource_class>: <resource_object>}
        """
        build_arguments = build_arguments or []
        environment = environment or {}

        logging.info('Running the %s phase...', ' '.join(build_arguments))

        env = os.environ.copy()
        env.update(environment)

        log_prefix = build_arguments[0] if build_arguments else 'build'
        self._run_build_script(build_arguments, env, log_prefix)

    def _run_build_script(self, build_arguments, env, log_prefix):
        sandbox_process.run_process(
            [
                'bash',
                os.path.join(self.checkout_path, 'build_all_ci.sh'),
            ] + build_arguments,
            log_prefix=log_prefix,
            environment=env,
        )

    def _publish(self, resources=None, resources_attrs=None, copy_resources=True):
        published = super(BaseQuasarImageBuildTask, self)._publish(
            resources=resources,
            resources_attrs=resources_attrs,
            copy_resources=copy_resources,
        )

        for resource in published.values():
            if isinstance(resource, qrt._QuasarOTABase):
                daemons = self.Parameters.quasar_daemons

                if daemons is not None and daemons.major_release_num:
                    resource.major_release_num = daemons.major_release_num

                if daemons is not None and daemons.minor_release_num:
                    resource.minor_release_num = daemons.minor_release_num

                resource.crc32_checksum = quasar_utils.get_resource_crc32(resource)

                if (
                        self.Context.arcanum_review_id != ctm.NotExists and
                        self.Context.is_patch_check == ctm.NotExists
                ):
                    resource.do_not_publish_me = True

                if self.Context.arcanum_review_id != ctm.NotExists:
                    resource.arcanum_review_id = self.Context.arcanum_review_id

                self._ota_images += [resource.id]

        return published

    def _check_input_resoruces(self):
        daemons = self.Parameters.quasar_daemons
        if daemons is not None:
            if daemons.quasar_platform != self.config.platform:
                raise sandbox_errors.TaskFailure('quasar_daemons and task platforms do not match')
            daemons_platform_revision = daemons.quasar_platform_revision
            platform_revision = self.Parameters.quasar_platform_revision
            if platform_revision and daemons_platform_revision != platform_revision:
                raise sandbox_errors.TaskFailure(
                    "Daemons resource revision attribute ({}) is not equals to expected revision ({})".format(
                        daemons_platform_revision, platform_revision
                    )
                )

    @staticmethod
    def _add_write_permissions(path):
        logging.debug("Adding write permissions for path '%s'", path)

        sandbox_process.run_process(
            "chmod +w -R {}".format(path),
            shell=True,
            log_prefix='chmod_plus_w',
        )

    def _place_daemons(self, factory=False):
        source_path = str(self.config.factory_daemons_source_path if factory else self.config.daemons_source_path)
        dest_path = str(self.config.factory_daemons_dest_path if factory else self.config.daemons_dest_path)
        logging.info("Copying %s daemons from '%s' to '%s'", 'factory' if factory else '', source_path, dest_path)

        sandbox_paths.remove_path(dest_path)
        sandbox_paths.copy_path(source_path, dest_path, symlinks=True)
        self._add_write_permissions(dest_path)

    def _place_resources(self):
        self._place_daemons()

    def _strip_binaries(self, path=None):
        """
        Strips placed binaries of debug symbols.
        Ignore errors as some files are non-strippable, but we dont know which ones.
        """
        target_path = path or self.config.daemons_dest_path

        try:
            sandbox_process.run_process(
                '%s * libs/* fluent-bit/*' % self.config.strip_tool_path,
                work_dir=str(target_path),
                shell=True,
                log_prefix='strip_daemons',
            )
        except sandbox_process.errors.SandboxSubprocessError:
            pass

    def _on_prepare(self):
        super(BaseQuasarImageBuildTask, self)._on_prepare()
        self._check_input_resoruces()
        self.config.load_package_config()
        self.config.load_factory_package_config()

    def _place_package_cache(self):
        """
        Places downloaded packages of Buildroot into source tree so they are
        not downloaded from Internet.
        """

        dest_path = self.config.buildroot_cache_dest_path
        dest_path.mkdir(mode=0o775, exist_ok=True)

        package_cache = sdk2.Resource.find(
            resource_type=qrt.QuasarBuildrootPkgCache,
            attrs={
                "quasar_platform": self.config.buldroot_cache_platform or self.config.platform
            }
        ).first()

        if package_cache is None:
            logging.info('Package cache not found.')
            return

        download_path = sdk2.path.Path(sdk2.ResourceData(package_cache).path)

        logging.info('Downloaded res %s to %s', package_cache, download_path)

        logger = quasar_utils.create_task_logger(self, 'package_cache.log', 'package_cache.placing')

        for source_entry in download_path.iterdir():
            dest_entry = dest_path.joinpath(source_entry.name)

            logger.debug('Copying %s to %s', source_entry, dest_entry)

            if source_entry.is_dir():
                shutil.copytree(str(source_entry), str(dest_entry), symlinks=True)
            elif source_entry.is_file():
                shutil.copy2(str(source_entry), str(dest_entry))
            else:
                continue
