import json
import logging
import os
import shutil
import subprocess

import sandbox.common.enum as sandbox_enum
import sandbox.common.errors as sandbox_errors
import sandbox.common.types.client as ctc
import sandbox.sdk2 as sdk2
import sandbox.projects.common.arcadia.sdk as arc_sdk

import sandbox.projects.quasar.build_types as build_types
import sandbox.projects.quasar.image_builds.base_image_build as base_image_build
import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.resource_types.tv as tv_resource_types
import sandbox.projects.quasar.utils as quasar_utils
import sandbox.projects.quasar.utils.vcs as vcs_utils


class QuasarBuildTvImageBase(
        base_image_build.BaseAndroidImageBuildTask,
        quasar_utils.YAVExportMixin,
):
    YAV_KEY_VAULT_OWNER = 'ANDROID_TV'
    YAV_SSH_PRIVATE_KEY_VAULT_NAME = 'robot-edi-ssh'
    YAV_SSH_LOGIN = 'robot-edi'

    class OutputArtifactType(sandbox_enum.Enum):
        ROM = None
        OTA = None
        UNIOTA = None

    class BuildConfig(base_image_build.BaseAndroidImageBuildTask.BuildConfig):
        ssh_private_key_vault_name = 'robot-edi-ssh'
        ssh_private_key_vault_owner = 'ANDROID_TV'
        default_vcs = vcs_utils.VCS.REPO
        repository_user = 'robot-edi'
        downloaded_artifacts_config = tv_resource_types.AndroidTvArtifactsConfig
        downloaded_artifacts = tv_resource_types.AndroidTvArtifacts

        @property
        def zip_file_path(self):
            return os.path.join(self.reports_path, 'VendorConfig.apk.zip')

        @property
        def checkout_path(self):
            return str(self.task.path('tv_repository'))

        @property
        def platform(self):
            raise NotImplementedError

        @property
        def reports_path(self):
            return str(self.task.path('reports'))

        @property
        def system_build_prop_path(self):
            return os.path.join(
                self._target_product_path,
                'system/build.prop'
            )

        @property
        def extra_build_prop_path(self):  # noqa
            return None

        @property
        def platform_build_prop_path(self):  # noqa
            return None

        @property
        def vendor_build_prop_path(self):  # noqa
            return None

        @property
        def error_logo_path(self):  # noqa
            return None

        @property
        def recovery_logo_path(self):  # noqa
            return None

        @property
        def uniota_path(self):  # noqa
            return None

    class Requirements(
            quasar_utils.YAVExportMixin.Requirements,
            base_image_build.BaseAndroidImageBuildTask.Requirements
    ):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.SSD
        disk_space = 300 * 1024  # 300 Gb
        cores = 16
        ram = 63 * 1024  # Mb

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(base_image_build.BaseAndroidImageBuildTask.Parameters):
        _container = quasar_utils.LastStableContainer(
            'Execution container', resource_type=tv_resource_types.AndroidTvLxcContainer)
        kill_timeout = 3 * 60 * 60
        suspend_on_exception = sdk2.parameters.Bool(
            'Suspend task for debug purposes when some error happens.', default=False)

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description='Key-value pairs of inspection numbers and local paths'
                        ' of projects relative to the repo\'s root')
        custom_manifest = sdk2.parameters.String('Arcadia url for manifest')

        secret_device_keys = sdk2.parameters.String(
            'Tv device keys yav secret', default='sec-01ekjdg241gw03p423knj3e5c2')

        apps_major_version = sdk2.parameters.String('Apps major version number. Usually branch number.')
        apps_minor_version = sdk2.parameters.String('Apps minor version number. Usually tag number in branch.')

        with sdk2.parameters.String('Build type', required=True, default=build_types.ImageBuildtype.USER) as build_type:
            for image_build_type in list(build_types.ImageBuildtype):
                build_type.values[image_build_type] = image_build_type
        build_ota = sdk2.parameters.Bool('Build OTA')

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def get_product_target(self):
        return self.target_config['target']

    @staticmethod
    def mkdir(path):
        if not os.path.exists(path):
            os.mkdir(path)
        elif not os.path.isdir(path):
            raise sandbox_errors.TaskFailure('{} should be directory, but it\'s not.'.format(path))

    def _build_target_report(self):
        self.mkdir(self.config.reports_path)
        self.extract_vendor_config()
        self.copy_images()
        self.copy_properties()

    def _on_execute(self):
        try:
            with sdk2.helpers.ProgressMeter('Checking out the repository'):
                self._checkout()
                logging.info('Checkout complete')
            with sdk2.helpers.ProgressMeter('Downloading prebuilt artifacts'):
                self._download_prebuilt()
                logging.info('Download prebuilts complete')
            with sdk2.helpers.ProgressMeter('Building AOSP'):
                self._build()
                logging.info('Build complete')
            with sdk2.helpers.ProgressMeter('Extracting tv target customized stuff from build results'):
                self._build_target_report()
                logging.info('Target report complete')
            with sdk2.helpers.ProgressMeter('Publishing results'):
                self._publish_result()
                logging.info('Results published')
        except Exception:  # noqa
            if self.Parameters.suspend_on_exception:
                logging.exception('We expected something like this, so we are suspending task for debug purpose.')
                self.suspend()
            else:
                raise

    @property
    def device_key_path(self):
        return os.path.join(self.checkout_path, 'device_key.pem')

    @property
    def rom_dir_name(self):
        '''
        Returns name used for rom dir and all images
        Can be used only after _on_prepare call of base class.
        '''
        rom_dir_name_list = [
            '{product_target}',
            '{platform_board}',
            '{repository_tag}',
            '{build_type}',
            '{build_number}',
        ]

        return '_'.join(rom_dir_name_list).format(  # noqa
            product_target=self.Parameters.build_product_target_name.replace('=', ''),
            platform_board=self.config.platform_board,
            repository_tag=self.repository_tag.replace('/', '_'),
            build_type=self.Parameters.build_type,
            build_number=self.id,
        ).upper()

    def prepare_versions(self):
        app_major_versions = set()
        app_minor_versions = set()
        self.apps_version = {  # noqa
            'major': 'none',
            'minor': 'none',
        }
        if self.Parameters.artifact_downloader_dependencies_json:
            for package_name, resource_id in self.Parameters.artifact_downloader_dependencies_json.items():
                resource = sdk2.Resource[resource_id]
                if resource.type == tv_resource_types.AndroidTvApp:
                    version_parts = resource.version_name.split('.')
                    if (
                            (
                                self.Parameters.apps_major_version and
                                self.Parameters.apps_major_version != version_parts[1]
                            ) or
                            (
                                self.Parameters.apps_minor_version and
                                self.Parameters.apps_minor_version != version_parts[2]
                            )
                    ):
                        raise sandbox_errors.TaskFailure('{} version mismatch {} != {}'.format(
                            package_name,
                            '.'.join((self.Parameters.apps_major_version, self.Parameters.apps_minor_version)),
                            '.'.join(version_parts[1:3])
                        ))
                    app_major_versions.add(version_parts[1])
                    app_minor_versions.add(version_parts[2])
            self.apps_version['major'] = list(app_major_versions)[0] if len(app_major_versions) == 1 else 'mix'
            self.apps_version['minor'] = list(app_minor_versions)[0] if len(app_minor_versions) == 1 else 'mix'

        logging.info('Found apps with version %s.%s', self.apps_version['major'], self.apps_version['minor'])
        logging.info('Verbose firmare version %s', self._determine_version())

    def prepare_tv_target_config(self):
        tv_platform = quasar_platform.TvPlatform(self.config.platform)
        with arc_sdk.mount_arc_path(
            self.Parameters.checkout_arcadia_from_url, use_arc_instead_of_aapi=True
        ) as mount_path:
            target_config_path = os.path.join(
                mount_path,
                'smart_devices/tv/platforms',
                tv_platform.odm,
                tv_platform.board_number,
                '{}.json'.format(self.Parameters.build_product_target_name),
            )
            if os.path.isfile(target_config_path):
                with open(target_config_path) as target_config_file:
                    target_config_data = target_config_file.read()
                    logging.info('Parsing target config\n%s', target_config_data)
                    self.target_config = json.loads(target_config_data)  # noqa

    def _on_prepare(self):
        with sdk2.helpers.ProgressMeter('Preparing image build stuff'):
            super(QuasarBuildTvImageBase, self)._on_prepare()
            if not os.path.exists(self.checkout_path):
                os.mkdir(self.checkout_path)

            logging.info('Ensure repository checkout path exists: %s', self.checkout_path)

            device_key_filepath = self.yav_export_file(
                str(self.Parameters.secret_device_keys), str(self.Parameters.secret_device_keys_key))
            shutil.copy(device_key_filepath, self.device_key_path)

            logging.info('Prepared device key')

            self.prepare_tv_target_config()
            self.prepare_versions()

            logging.info('Check rom dir name before build - %s', self.rom_dir_name)

    def _on_success(self):
        pass

    def unzip_from_vendor_config(self, resource_paths):
        unzip_cmd_list = [
            'unzip',
            '-j',
            self.config.zip_file_path,
            '-d', self.config.reports_path,
        ]
        unzip_cmd_list.extend(resource_paths)
        logging.info('Running %s', ' '.join(unzip_cmd_list))
        self.run_command_with_log(unzip_cmd_list, 'unzip')

    def extract_vendor_config(self):
        logging.info('Extract VendorConfig.apk')
        shutil.copy(self.config.vendor_config_path, self.config.zip_file_path)
        self.unzip_from_vendor_config([
            'res/drawable-xxhdpi-v4/suw_rc_image.png',
            'res/drawable-xxhdpi-v4/suw_rc_image_alice.png',
            'res/drawable/suw_rc_pairing_image.png',
            'res/drawable/suw_rc_pairing_image_alice.png',
        ])
        shutil.move(self.config.zip_file_path, os.path.join(self.config.reports_path, 'VendorConfig.apk'))

    def _determine_version(self):
        """
        :returns: a `str` version for this build.
        Version is defined as
        `f_1.f_2.a_1.a_2.task_id.build_date[.repository_tag].fw_build_type.apps_build_type[.factory_suffix]`,
        where:
            * f_1, f_2 - firmware repo version
            * a_1, a_2 - apps repo version (a_1 is branch number, a_2 is tag number)
            * task_id - this task id,
            * build_date - date of the build in the form YYYYMMDD
            * fw_build_type - build type of firmware (USER/ENG/USERDEBUG)
            * apps_build_type - build type of Android apps (USER/PRESTABLE/USERDEBUG)
            * factory_suffix - optional suffix for factory firmware
        """

        if '-' in self.repository_tag:
            version_part = self.repository_tag.split('-', 1)[1]
            version_numbers = version_part.split('.')
            if len(version_numbers) > 1:
                f_version = [version_numbers[0], version_numbers[1]]
            else:
                f_version = ['0', version_numbers[0]]
        else:
            f_version = ['0', '0']

        a_version = [str(self.apps_version['major']), str(self.apps_version['minor'])]

        suffix = [str(self.id), self.created.strftime('%Y%m%d')]

        if self.Parameters.build_type != self.config.release_build_type:
            suffix.append(str(self.Parameters.build_type).upper())

        if 'factory' in self.repository_tag:
            suffix.append('FACTORY')

        version = '.'.join(f_version + a_version + suffix)

        return version

    def publish_image(self, resources, resource_attrs):
        published = self._publish(resources, resource_attrs)
        for resource in published.values():
            resource.crc32_checksum = quasar_utils.get_resource_crc32(resource)

    def run_command_with_log(self, command_list, log_name, env=None):
        with sdk2.helpers.ProcessLog(self, logger=log_name) as pl:
            logging.info('Running %s', ' '.join(command_list))
            subprocess.check_call(
                command_list,
                stdout=pl.stdout,
                stderr=pl.stdout,
                env=env,
            )

    def copy_images(self):
        logging.info('Copy images')
        shutil.copy(self.config.boot_logo_path, self.config.reports_path)
        if self.config.error_logo_path:
            shutil.copy(self.config.error_logo_path, self.config.reports_path)
        if self.config.recovery_logo_path:
            shutil.copy(self.config.recovery_logo_path, self.config.reports_path)

    def copy_properties(self):
        shutil.copy(
            self.config.system_build_prop_path,
            os.path.join(self.config.reports_path, 'system_build.prop'))
        if self.config.platform_build_prop_path:
            shutil.copy(
                self.config.platform_build_prop_path,
                os.path.join(self.config.reports_path, 'platform_build.prop'))
        if self.config.vendor_build_prop_path:
            shutil.copy(
                self.config.vendor_build_prop_path,
                os.path.join(self.config.reports_path, 'vendor_build.prop'))
        if self.config.extra_build_prop_path:
            shutil.copy(
                self.config.extra_build_prop_path,
                os.path.join(self.config.reports_path, 'extra_build.prop'))

    def get_output_artifact_path(self, artifact_type=None):
        if artifact_type is None:
            return str(self.path(self.rom_dir_name))
        else:
            return os.path.join(
                str(self.path(self.rom_dir_name)),
                '{}-{}.zip'.format(self.rom_dir_name, str(artifact_type)),
            )

    def _publish_result(self):
        supported_target = {
            'brand': self.target_config['props']['brand'],
        }
        # temporary ugly hack, remove after full switch to model targeting
        if self.config.platform == quasar_platform.Platform.GOYA:
            supported_target['model'] = self.target_config['props']['device']
        else:
            supported_target['product'] = self.target_config['props']['device']

        target_resource_attrs = {
            'tv_target': self.Parameters.build_product_target_name,
            'supported_targets': json.dumps([supported_target]),
        }

        report_resource_attrs = {
            'platform': quasar_platform.TvPlatform(self.config.platform).quasar_platform,
            'branch': self.repository_tag,
        }
        target_report_resource_attrs = dict(report_resource_attrs)
        target_report_resource_attrs.update(target_resource_attrs)
        if os.path.isdir(self.config.reports_path):
            self._publish(
                resources={tv_resource_types.AndroidTvReport: self.config.reports_path},
                resources_attrs=target_report_resource_attrs,
            )

        image_resource_attrs = {
            'buildtype': self.Parameters.build_type,
            'version': self.id,
            'verbose_version': self._determine_version(),
        }
        target_image_resource_attrs = dict(image_resource_attrs)
        target_image_resource_attrs.update(target_resource_attrs)

        rom_dir_path = self.get_output_artifact_path()
        logging.info('Rom dir - %s', rom_dir_path)
        os.mkdir(rom_dir_path)

        if not self.target_config.get('uniota', False):
            self.fill_rom_dir(rom_dir_path)

            rom_filepath = self.get_output_artifact_path(self.OutputArtifactType.ROM)
            with sdk2.helpers.ProcessLog(self, logger='zip') as pl:
                current_dir = os.getcwd()
                os.chdir(os.path.dirname(rom_dir_path))
                subprocess.check_call(
                    ['zip', '-r', rom_filepath, self.rom_dir_name], stdout=pl.stdout, stderr=pl.stderr)
                os.chdir(current_dir)

            self.publish_image(
                resources={self.config.image_resource_type: rom_filepath},
                resource_attrs=target_image_resource_attrs,
            )

        if self.Parameters.build_ota:
            if self.target_config.get('uniota', False):
                uniota_filepath = self.get_output_artifact_path(self.OutputArtifactType.UNIOTA)
                shutil.move(self.config.uniota_path, uniota_filepath)
                uniota_resource_attrs = dict(target_image_resource_attrs)
                uniota_resource_attrs.update({
                    'supported_targets': json.dumps(self.Parameters.uniota_supported_targets),
                })
                self.publish_image(
                    resources={self.config.uniota_resource_type: uniota_filepath},
                    resource_attrs=uniota_resource_attrs,
                )
            else:
                ota_filepath = self.get_output_artifact_path(self.OutputArtifactType.OTA)
                shutil.move(self.config.ota_path, ota_filepath)
                self.publish_image(
                    resources={self.config.ota_resource_type: ota_filepath},
                    resource_attrs=target_image_resource_attrs,
                )
