import os
import time
import pwd
import multiprocessing
import logging
import subprocess

import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.image_builds.base_image_build.tv as tv_base


class QuasarBuildTvImageWithDockerBase(tv_base.QuasarBuildTvImageBase):
    class Parameters(tv_base.QuasarBuildTvImageBase.Parameters):
        docker_image = sdk2.parameters.Resource('Docker image used for build')

    @property
    def docker_name(self):
        return ':'.join((self.config.docker_img_name, self.config.docker_tag))

    def get_mount_specs(self, home_dir):
        self.mkdir(os.path.join(home_dir, '.ccache'))
        self.mkdir(os.path.join(home_dir, '.ssh'))

        stat_info = os.stat('.')
        current_user_id = os.getuid()
        if stat_info.st_uid != current_user_id:
            raise Exception('You must be the owner of work dir')

        if not os.path.isdir('.repo'):
            raise Exception('You must run this script in repository root dir')

        mount_specs = [
            '-v', '{home_dir}/.gitconfig:/home/{docker_user}/.gitconfig',
            '-v', '{home_dir}/.ssh:/home/{docker_user}/.ssh',
            '-v', '{home_dir}/.ccache:/home/{docker_user}/.ccache',
        ]
        for container_path in self.config.docker_code_paths:
            mount_specs.extend(['-v', '{}:{}'.format(self.checkout_path, container_path)])
        if os.path.isdir('/mirror'):
            mount_specs.extend(['-v', '{}:/mirror'.format(os.path.realpath('/mirror'))])

        return [mount_spec.format(home_dir=home_dir, docker_user=self.config.docker_user) for mount_spec in mount_specs]

    def get_container_name(self):
        code_dir_basename = os.path.basename(self.checkout_path)
        current_time = time.strftime('%Y%m%d%H%M%S')
        return '{}-{}-{}'.format(pwd.getpwuid(os.getuid())[0], current_time, code_dir_basename[:48])

    @staticmethod
    def get_cpu_count():
        return multiprocessing.cpu_count() - 2

    def docker_exec_bash_script(self, container_name, script, stage_name):
        docker_exec_cmd = 'docker exec {} bash -c "{}"'.format(container_name, script)
        logging.info('Running "%s"', docker_exec_cmd)
        with sdk2.helpers.ProcessLog(self, logger='docker_exec_' + stage_name) as pl:
            subprocess.check_call(docker_exec_cmd, stdout=pl.stdout, stderr=pl.stdout, shell=True)

    def docker_run(self, container_name):
        docker_image_path = str(sdk2.ResourceData(self.Parameters.docker_image).path)
        if docker_image_path.endswith('.zip'):
            task_dir = str(self.path())
            self.run_command_with_log(['unzip', docker_image_path, '-d', task_dir], 'unzip_docker_image')
            docker_image_path = os.path.join(task_dir, os.path.basename(docker_image_path.replace('.zip', '.tar')))
        docker_load_list = ['docker', 'load', '--input', docker_image_path]
        self.run_command_with_log(docker_load_list, 'docker_load')
        with sdk2.helpers.ProcessLog(self, logger='docker_images') as pl:
            docker_images_cmd_list = ['docker', 'images']
            logging.info('Running %s', ' '.join(docker_images_cmd_list))
            docker_images = subprocess.check_output(docker_images_cmd_list, stderr=pl.stdout)
        logging.info(docker_images)
        filtered_images = list([
            image
            for image in docker_images.split('\n')
            if self.config.docker_img_name in image and self.config.docker_tag in image
        ])
        if not filtered_images:
            self.run_command_with_log(['docker', 'tag', self.docker_name], 'docker_tag')

        home_dir = os.path.expanduser('~')
        mount_specs = self.get_mount_specs(home_dir)

        docker_run_list = [
            'docker', 'run',
            '-u', 'root',
            '--rm',
            '--name', container_name,
            '-dit',
            '--cpuset-cpus', '1-{}'.format(self.get_cpu_count()),
            '-h', self.config.platform_board + '-' + os.path.basename(self.checkout_path),
        ]
        docker_run_list.extend(mount_specs)
        docker_run_list.append(self.docker_name)
        self.run_command_with_log(docker_run_list, 'docker_run')

        os_user_id = os.getuid()
        os_group_id = os.getuid()
        self.docker_exec_bash_script(
            container_name, 'usermod -u {} {}'.format(os_user_id, self.config.docker_user), 'setup_user')
        self.docker_exec_bash_script(
            container_name, 'groupmod -g {} {}'.format(os_group_id, self.config.docker_user), 'setup_user')

    @staticmethod
    def choose_target():
        '''
            Function reserved for cvte builds. Idea is to select build target by editing repository,
            rather than by providing some parameter to docker command.
        '''
        pass

    def _build(self):
        os.chdir(self.checkout_path)

        self.choose_target()
        container_name = self.get_container_name()
        self.docker_run(container_name)

        image_build_script = 'echo -e \\"{docker_command}\\" | su -l {docker_user}; exit \\$?'.format(
            docker_command=' && '.join(self.docker_command_list),
            docker_user=self.config.docker_user,
        )
        self.docker_exec_bash_script(container_name, image_build_script, 'image_build')
