import os
import logging
import shutil

import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.build_types as build_types
import sandbox.projects.quasar.image_builds.base_image_build.tv_docker as tv_docker_base


class QuasarBuildTvMtkImageBase(tv_docker_base.QuasarBuildTvImageWithDockerBase):
    class BuildConfig(tv_docker_base.QuasarBuildTvImageWithDockerBase.BuildConfig):
        android_version = 9

        @property
        def _mediatek_linux_output(self):
            return os.path.join(self._out_path, 'mediatek_linux/output')

        @property
        def _tvconfig_path(self):
            return os.path.join(self._mediatek_linux_output, 'mediatek/{}/rel/tvconfig'.format(self.mediatek_device))

        @property
        def _target_product_path(self):
            return os.path.join(self._out_path, 'target/product/{}'.format(self.mediatek_device))

        @property
        def vendor_build_prop_path(self):
            return os.path.join(
                self._target_product_path,
                'vendor/build.prop'
            )

        @property
        def boot_logo_path(self):
            return os.path.join(
                self._tvconfig_path,
                'boot0.jpg'
            )

        @property
        def error_logo_path(self):
            return os.path.join(
                self._tvconfig_path,
                'boot_error.jpg'
            )

        @property
        def recovery_logo_path(self):
            return os.path.join(
                self._tvconfig_path,
                'boot_recovery.jpg'
            )

    class Parameters(tv_docker_base.QuasarBuildTvImageWithDockerBase.Parameters):
        uniota_supported_targets = sdk2.parameters.JSON(
            'Matching targets for uniota build: [{"brand": "brand", "product": "device"}]',
            default=[]
        )


class QuasarBuildTvMtkCvImageBase(QuasarBuildTvMtkImageBase):
    class Parameters(QuasarBuildTvMtkImageBase.Parameters):
        docker_image = sdk2.parameters.Resource('Docker image used for build', default=2240136380)
        kill_timeout = 5 * 60 * 60
        ota_accept_device = sdk2.parameters.String(
            'Specify "device" build prop value of tv if you want to build ota with'
            'different "device" value which will not be rejected during update process.')

    class BuildConfig(QuasarBuildTvMtkImageBase.BuildConfig):
        default_repository_url = 'ssh://gerrit.yandex-team.ru/mtk9632/manifest'
        docker_img_name = 'ubuntu'
        docker_tag = 'aosp_fix'
        docker_user = 'mrfirst'
        docker_code_paths = ('/mnt',)
        yandex_prebuilt_repo_path = 'vendor/yandex/prebuilt'

        @property
        def _out_path(self):
            return os.path.join(self.checkout_path, 'out')

        @property
        def rom_path(self):
            return os.path.join(
                self._out_path,
                'images/android_{}/{}/CtvUpgrade.bin'.format(self.android_version, self.mediatek_device)
            )

        @property
        def ota_path(self):
            if self.task.Parameters.build_type == build_types.ImageBuildtype.USERDEBUG:
                ota_path_template = os.path.join(self._mediatek_linux_output, '{}-ota_update.zip')
            else:
                ota_path_template = os.path.join(self._mediatek_linux_output, '{}-signed-ota_update.zip')

            return ota_path_template.format(self.mediatek_device)

        @property
        def uniota_path(self):
            return os.path.join(self._mediatek_linux_output, '{}-uniota_update.zip'.format(self.mediatek_device))

        @property
        def mboot_bin_path(self):
            return os.path.join(self._out_path, 'images/android_{}/{}/mboot.bin'.format(
                self.android_version, self.mediatek_device))

        @property
        def romboot_bin_path(self):
            return os.path.join(self._out_path, 'images/android_{}/{}/rom_emmc_boot.bin'.format(
                self.android_version, self.mediatek_device))

        @property
        def platform_build_prop_path(self):
            return os.path.join(self._tvconfig_path, 'config/ctvbuild.prop')

        @property
        def vendor_config_path(self):
            return os.path.join(self._tvconfig_path, 'android/app/VendorConfig/VendorConfig.apk')

    def get_customer(self):
        customer_dict = {
            'ARTEL': 'ARTEL',
            'CHANGHONG': 'ChangHong',
            'CTV': 'CTV',
            'DCZN': 'DCZN',
            'EXPRESSLUCK': 'Expressluck',
            'JINPIN': 'Jinpin',
            'KVANT': 'Kvant',
            'PKV': 'LERAN',
            'UNIOTA': 'UniOTA',
            'YANDEX': 'Yandex',
            'ZHAOCHI': 'ZHAOCHI',
        }
        for customer_substring, customer_dirname in sorted(customer_dict.items()):
            logging.debug('%s, %s', customer_substring, self.Parameters.build_product_target_name)
            if customer_substring in self.Parameters.build_product_target_name:
                return customer_dirname

        raise Exception('Can not parse the customer')

    @property
    def ext_make_args(self):  # noqa
        return []

    @property
    def docker_command_list(self):
        customer = self.get_customer()
        make_args = '-j DEVICE_KEY=/mnt/{} RLS_CUSTOM_BUILD=true'.format(os.path.basename(self.device_key_path))
        if self.Parameters.build_ota:
            make_args += ' ENABLE_OTA=true'
            if self.Parameters.build_type == build_types.ImageBuildtype.USER:
                make_args += ' SIGN_IMG_SUPPORT=true'
            if self.Parameters.ota_accept_device:
                make_args += ' OTA_LEGACY_DEVICE=' + self.Parameters.ota_accept_device
        make_args += ' ' + ' '.join(self.ext_make_args)
        make_args += ' mtk_build'
        build_env = {
            'TS_BUILD_COUNT': str(self.id),
            'BRANCH_NAME_DEF': self.repository_tag,
            'PLATFORM_BOARD_NAME': self.config.platform_board,
        }

        lunch_string = 'ctv_lunch build/build_config/{customer_name}/{build_product_target}.txt {build_variant}'
        docker_command_list = list(['export {}={}'.format(key, value) for key, value in build_env.items()])
        docker_command_list.extend([
            'cd /mnt',
            'source build/ctv_envsetup.sh',
            lunch_string.format(
                customer_name=customer,
                build_product_target=self.get_product_target(),
                build_variant=self.Parameters.build_type),
            'make ' + make_args,
        ])
        return docker_command_list

    def fill_rom_dir(self, rom_dir_path):
        shutil.copy(self.config.rom_path, rom_dir_path)
        shutil.copy(self.config.romboot_bin_path, rom_dir_path)
        shutil.copy(self.config.mboot_bin_path, rom_dir_path)


class QuasarBuildTvMtkCvteImageBase(QuasarBuildTvMtkImageBase):
    class Parameters(QuasarBuildTvMtkImageBase.Parameters):
        docker_image = sdk2.parameters.Resource('Docker image used for build', default=1902999534)
        kill_timeout = 4 * 60 * 60

    class BuildConfig(QuasarBuildTvMtkImageBase.BuildConfig):
        default_repository_url = 'ssh://gerrit.yandex-team.ru/cvte9632/manifest'
        docker_img_name = 'android_mm_build'
        docker_tag = 'latest'
        docker_user = 'builder'
        docker_code_paths = ('/home/code', '/home/builder/code')
        customer = 'CUSTOMER_YANDEX_UNI'
        yandex_prebuilt_repo_path = 'android/vendor/yandex/prebuilt'
        mboot_bin_path = 'bin/mboot.bin'
        romboot_bin_path = 'bin/rom_emmc_boot.bin'

        @property
        def _out_path(self):
            return os.path.join(self.checkout_path, 'android/out')

        @property
        def ocs_xml_path(self):
            for node in os.listdir(os.path.join(self.checkout_path, 'bin')):
                if node.endswith('.xml'):
                    return os.path.join(self.checkout_path, 'bin', node)

        @property
        def rom_path(self):
            for node in os.listdir(os.path.join(self.checkout_path, 'bin')):
                if node.startswith(self.rom_filename_prefix) and node.endswith('.bin'):
                    return os.path.join(self.checkout_path, 'bin', node)

        @property
        def ota_path(self):
            for node in os.listdir(os.path.join(self.checkout_path, 'bin')):
                if node.startswith('OTA') and node.endswith('.zip'):
                    return os.path.join(self.checkout_path, 'bin', node)

        @property
        def uniota_path(self):
            for node in os.listdir(self._mediatek_linux_output):
                if (
                        node.startswith(self.mediatek_device.split('_')[0]) and
                        node.endswith('-uniota_update.zip')
                ):
                    return os.path.join(self._mediatek_linux_output, node)

        @property
        def platform_build_prop_path(self):
            return os.path.join(
                self._out_path,
                'target/product/{}/tvconfig/build.prop'.format(self.mediatek_device)
            )

        @property
        def vendor_config_path(self):
            return os.path.join(
                self._out_path,
                'target/product/{}/tvconfig/app/VendorConfig/VendorConfig.apk'.format(self.mediatek_device)
            )

    def choose_target(self):
        choose_customer_tool_path = os.path.join(self.checkout_path, 'customers/choose-target.sh')
        choose_target_list = [choose_customer_tool_path, self.config.customer, self.get_product_target()]
        self.run_command_with_log(choose_target_list, 'choose_target')

    @property
    def docker_command_list(self):
        if self.Parameters.build_ota:
            make_args = 'ota '
        else:
            make_args = 'ocs '
        make_args_template = (
            'MODE={build_type} '
            'MODEL={build_product_target} '
            'BRANCH_NAME_DEF={repository_tag} '
            'DEVICE_KEY=\\$(readlink -f device_key.pem)'
        )
        make_args += make_args_template.format(
            build_type=self.Parameters.build_type,
            build_product_target=self.get_product_target(),
            repository_tag=self.repository_tag
        )
        if self.Parameters.build_type != build_types.ImageBuildtype.USER:
            make_args += ' ADB=on'

        build_env = {
            'TS_BUILD_COUNT': str(self.id),
            'PLATFORM_BOARD_NAME': self.config.platform_board,
        }

        docker_command_list = list(['export {}={}'.format(key, value) for key, value in build_env.items()])
        docker_command_list.append('make ' + make_args)

        return docker_command_list

    def fill_rom_dir(self, rom_dir_path):
        shutil.copy(self.config.rom_path, os.path.join(rom_dir_path, self.config.rom_filename_prefix + '_sos.bin'))
        shutil.copy(self.config.romboot_bin_path, rom_dir_path)
        shutil.copy(self.config.mboot_bin_path, rom_dir_path)
        shutil.copy(self.config.ocs_xml_path, rom_dir_path)


class QuasarBuildTvMtkCvteAndroid11ImageBase(QuasarBuildTvMtkCvteImageBase):
    class BuildConfig(QuasarBuildTvMtkCvteImageBase.BuildConfig):
        default_repository_url = 'ssh://gerrit.yandex-team.ru/cvte9256/manifest'
        native_emmc_bin_path = 'bin/NativeEmmc.bin'

        @property
        def platform_build_prop_path(self):
            return os.path.join(
                self._tvconfig_path,
                'cvte.prop'
            )

        @property
        def vendor_config_path(self):
            return os.path.join(
                self._out_path,
                'target/product/{}/system/app/VendorConfig/VendorConfig.apk'.format(self.mediatek_device)
            )

        @property
        def _common_misc_path(self):
            return os.path.join(
                self._mediatek_linux_output,
                'mediatek/{}/rel/obj/sys_build/output/rel/cusdata/bsp/common/misc'.format(self.mediatek_device)
            )

        @property
        def boot_logo_path(self):
            return os.path.join(
                self._common_misc_path,
                'boot0.jpg'
            )

        @property
        def error_logo_path(self):
            return os.path.join(
                self._common_misc_path,
                'boot_error.jpg'
            )

        @property
        def recovery_logo_path(self):
            return os.path.join(
                self._common_misc_path,
                'boot_recovery.jpg'
            )

        @property
        def extra_build_prop_path(self):
            return os.path.join(
                self._target_product_path,
                'obj/ETC/system_build_prop_intermediates/build.prop'
            )

    def fill_rom_dir(self, rom_dir_path):
        super(QuasarBuildTvMtkCvteAndroid11ImageBase, self).fill_rom_dir(rom_dir_path)
        if not self.Parameters.build_ota:
            # only factory builds have NativeEmmc.bin, ota builds do not
            shutil.copy(self.config.native_emmc_bin_path, rom_dir_path)
