import os
import logging
import shutil

import sandbox.projects.quasar.image_builds.base_image_build.tv as tv_base


class QuasarBuildTvRtkImageBase(tv_base.QuasarBuildTvImageBase):
    class BuildConfig(tv_base.QuasarBuildTvImageBase.BuildConfig):
        android_version = 7
        error_logo_path = None
        recovery_logo_path = None
        yandex_prebuilt_repo_path = 'platform/kernel/android/nougat/vendor/yandex/prebuilt'

        @property
        def rom_path(self):
            return os.path.join(
                self.checkout_path,
                'platform/image_file_creator/install.img'
            )

        @property
        def _target_product_path(self):
            return os.path.join(
                self.checkout_path,
                'platform/kernel/android/nougat/out/target/product',
                self.task.target_config['target'],
            )

        @property
        def ota_path(self):
            for node in os.listdir(self._target_product_path):
                if node.startswith(self.task.target_config['target']) and node.endswith('.zip'):
                    return os.path.join(self._target_product_path, node)

        @property
        def boot_logo_path(self):
            return os.path.join(
                self.checkout_path,
                'platform/kernel/android/nougat/device',
                'realtek/RealtekATV/customization_related/defaultLogo/bootfile.raw'
            )

        @property
        def vendor_config_path(self):
            return os.path.join(
                self._target_product_path,
                'system/app/VendorConfig/VendorConfig.apk'
            )

    def _on_prepare(self):
        super(QuasarBuildTvRtkImageBase, self)._on_prepare()
        self.target_config = {'props': {}}  # noqa
        split_product = self.Parameters.build_product_target_name.split('-')
        # remove "=" before and after panel number
        self.target_config['panel'] = split_product[0].strip('=')

        if len(split_product) < 4:
            raise RuntimeError(
                'Cannot use product {}. Wrong format. '.format(self.Parameters.build_product_target_name) +
                'It must contain 4+ words separated by spaces e.g. ' +
                '"=240=-DEXP-50in-U50E9100Q-CHOT-CV500U1-T01"')

        # take model from "=240= DEXP 50in U50E9100Q CHOT CV500U1-T01" like string
        self.target_config['target'] = split_product[3]

        # verify that panel number is numeric
        if not self.target_config['panel'].isdigit():
            raise RuntimeError('Invalid panel number: {}'.format(self.target_config['panel']))

    def _build(self):
        os.chdir(os.path.join(self.checkout_path, 'platform'))
        bootcode_tmp_path = 'bootcode/tmp'
        if os.path.isdir(bootcode_tmp_path):
            shutil.rmtree(bootcode_tmp_path)
        logging.info('Select panel')
        self.run_command_with_log(['./customerselect.sh', self.target_config['panel']], 'customerselect')
        logging.info('Build bootcode')
        self.run_command_with_log(['./build_bootcode_2871_1_5G.sh'], 'build_bootcode')
        logging.info('Build android')
        build_command_list = [
            './build_android_all_2871_1_5G.sh',
            '--product-target', self.target_config['target'],
            '--build-variant', self.Parameters.build_type,
            '--build-ota', 'y' if self.Parameters.build_ota else 'n',
            '--branch-name', self.repository_tag,
            '--board-name', self.config.platform_board.split('_')[1],
            '--platform-board-name', self.config.platform_board,
            '--device-key', self.device_key_path]

        if self.Parameters.build_ota_with_bootcode:
            build_command_list.append('--ota-with-bootcode')

        build_env = dict(os.environ)
        build_env.update({
            'BUILD_NUMBER': str(self.id),
        })

        self.run_command_with_log(
            build_command_list,
            'build_android',
            env=build_env,
        )
        logging.info('Create install image and bootloader bin')
        self.run_command_with_log(
            [
                './installimage.sh',
                '--install-bootloader', '1',
                '--build-variant', self.Parameters.build_type
            ],
            'install_image_with_bootloader',
        )

        os.chdir(str(self.path()))

        with open(self.config.system_build_prop_path) as system_build_prop_file:
            system_build_props = system_build_prop_file.readlines()

        for line in system_build_props:
            prop_name_value = line.strip('\n').split('=')
            if prop_name_value[0] == 'ro.product.brand':
                self.target_config['props']['brand'] = prop_name_value[1]
            if prop_name_value[0] == 'ro.product.device':
                self.target_config['props']['device'] = prop_name_value[1]

    def fill_rom_dir(self, rom_dir_path):
        shutil.copy(self.config.rom_path, rom_dir_path)
