import logging
import json

import sandbox.sdk2 as sdk2
import sandbox.common.errors as sandbox_errors
import sandbox.projects.quasar.image_builds.mt9632_cv as base
import sandbox.projects.quasar.resource_types.tv as qrt
import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.platform.goya as goya
import sandbox.projects.quasar.build_types as build_types


class QuasarBuildGoyaImage(base.QuasarBuildMt9632CvImage):
    """
    Build full image for quasar device and publish it as a resource
    """
    class Parameters(base.QuasarBuildMt9632CvImage.Parameters):
        with sdk2.parameters.String(
            'Product target', required=True, default=goya.target_configs.keys()[0]
        ) as build_product_target_name:
            for name in goya.target_configs.keys():
                build_product_target_name.values[name] = name
        build_factory = sdk2.parameters.Bool('Build factory (mutually exclusive with Build OTA)', default=False)
        secret_device_keys_key = sdk2.parameters.String(
            'Yav secret key with device public key inside', default='goya.pem')

    class BuildConfig(base.QuasarBuildMt9632CvImage.BuildConfig):
        platform = quasar_platform.Platform.GOYA
        image_resource_type = qrt.AndroidTvGoyaImage
        ota_resource_type = qrt.AndroidTvGoyaOtaImage
        uniota_resource_type = qrt.AndroidTvGoyaUniotaImage

    def prepare_tv_target_config(self):
        self.target_config = goya.target_configs[self.Parameters.build_product_target_name]  # noqa
        logging.info('Crafted target config\n%s', json.dumps(self.target_config))

    def _on_prepare(self):
        super(QuasarBuildGoyaImage, self)._on_prepare()
        if self.Parameters.build_ota and self.Parameters.build_factory:
            raise sandbox_errors.TaskFailure(
                'Build OTA and Build factory are mutually exclusive and can\'t be checked together.')

    @property
    def ext_make_args(self):
        args = []
        if self.Parameters.build_factory:
            args.extend(['ENABLE_OTA=true', 'FACTORY_BUILD=true'])
            if self.Parameters.build_type == build_types.ImageBuildtype.USER:
                args.append('SIGN_IMG_SUPPORT=true')
        return args
