import logging
import os
import shutil

import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.image_builds.base_image_build.tv_docker as tv_docker_base
import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.platform.hisi351_cvte as hisi351_cvte
import sandbox.projects.quasar.resource_types.tv as qrt


class QuasarBuildHisi351CvteImage(tv_docker_base.QuasarBuildTvImageWithDockerBase):
    """
    Build full image for quasar device and publish it as a resource
    """
    class Parameters(tv_docker_base.QuasarBuildTvImageWithDockerBase.Parameters):
        with sdk2.parameters.String('Product target', required=True) as build_product_target_name:
            for key in sorted(hisi351_cvte.target_names):
                build_product_target_name.values[key] = None
        secret_device_keys_key = sdk2.parameters.String(
            'Yav secret key with device public key inside', default='hisi351.pem')
        docker_image = sdk2.parameters.Resource('Docker image used for build', default=1903124533)
        artifact_downloader_force_vendor_modules = sdk2.parameters.Bool(
            'Put applications to vendor partition where applicable', default=True)

    class BuildConfig(tv_docker_base.QuasarBuildTvImageWithDockerBase.BuildConfig):
        platform = quasar_platform.Platform.YANDEX_TV_HISI351_CVTE
        platform_board = 'cvte_hisi351'
        image_resource_type = qrt.AndroidTvHisi351CvteImage
        ota_resource_type = qrt.AndroidTvHisi351CvteOtaImage
        default_repository_url = 'ssh://gerrit.yandex-team.ru/cvte351/manifest'
        yandex_prebuilt_repo_path = 'Android/vendor/yandex/prebuilt'
        docker_img_name = 'registry.gz.cvte.cn/tv/android_hisi_build'
        docker_tag = '35x_v1'
        docker_user = 'builder'
        docker_code_paths = ('/home/code', '/home/builder/code')

        @property
        def rom_path(self):
            for node in os.listdir(os.path.join(self.checkout_path, 'bin')):
                if node.endswith('.bin'):
                    return os.path.join(self.checkout_path, 'bin', node)

        @property
        def _target_product_path(self):
            return os.path.join(self.checkout_path, 'Android/out/target/product/Hi3751V350')

        @property
        def ota_path(self):
            return os.path.join(self._target_product_path, 'Emmc/update.zip')

        @property
        def platform_build_prop_path(self):
            return os.path.join(self._target_product_path, 'vendor/cvte.prop')

        @property
        def vendor_config_path(self):
            return os.path.join(self._target_product_path, 'system/app/VendorConfig/VendorConfig.apk')

        @property
        def boot_logo_path(self):
            return os.path.join(self._target_product_path, 'vendor/etc/logo.jpg')

    def get_customer(self):
        customer_list = [
            'KVANT',
            'CHANGHONG',
            'YINGSHI',
            'MINGXIN',
            'HUIDI',
            'JINPIN',
            'CAIXUN',
            'YANDEX',
        ]
        for customer_name in sorted(customer_list):
            logging.debug('%s, %s', customer_name, self.Parameters.build_product_target_name)
            if customer_name in self.Parameters.build_product_target_name:
                return 'CUSTOMER_' + customer_name

        raise Exception('Can not parse the customer')

    def choose_target(self):
        choose_customer_tool_path = os.path.join(self.checkout_path, 'customers/choose-target.sh')
        choose_target_list = [choose_customer_tool_path, self.get_customer(), self.get_product_target()]
        self.run_command_with_log(choose_target_list, 'choose_target')

    @property
    def docker_command_list(self):
        build_env = {
            'TS_BUILD_COUNT': str(self.id),
            'PLATFORM_BOARD_NAME': self.config.platform_board,
        }
        docker_command_list = list(['export {}={}'.format(key, value) for key, value in build_env.items()])

        make_cmd_template = (
            'make ocs '
            'MODE={build_type} '
            'MODEL={build_product_target} '
            'BRANCH_NAME={repository_tag} '
            'DEVICE_KEY=\\$(readlink -f device_key.pem)'
        )
        make_cmd = make_cmd_template.format(
            build_type=self.Parameters.build_type,
            build_product_target=self.get_product_target(),
            repository_tag=self.repository_tag,
        )
        docker_command_list.append(make_cmd)

        return docker_command_list

    def fill_rom_dir(self, rom_dir_path):
        shutil.copy(self.config.rom_path, os.path.join(rom_dir_path, 'allupgrade_V351_sos.bin'))
