import os
import shutil

import sandbox.sdk2 as sdk2

import sandbox.projects.quasar.build_types as build_types
import sandbox.projects.quasar.image_builds.base_image_build.tv as base
import sandbox.projects.quasar.platform as quasar_platform
import sandbox.projects.quasar.platform.rt2842_hikeen as rt2842_hikeen
import sandbox.projects.quasar.resource_types.tv as qrt


class QuasarBuildRt2842HikeenImage(base.QuasarBuildTvImageBase):
    """
    Build full image for quasar device and publish it as a resource
    """
    class Parameters(base.QuasarBuildTvImageBase.Parameters):
        with sdk2.parameters.String('Product target', required=True) as build_product_target_name:
            for key in sorted(rt2842_hikeen.target_names):
                build_product_target_name.values[key] = None
        secret_device_keys_key = sdk2.parameters.String(
            'Yav secret key with device public key inside', default='yandex_tv_rt2842_hikeen.pem')

    class BuildConfig(base.QuasarBuildTvImageBase.BuildConfig):
        platform = quasar_platform.Platform.YANDEX_TV_RT2842_HIKEEN
        platform_board = 'hikeen_rt2842'
        image_resource_type = qrt.AndroidTvRt2842HikeenImage
        ota_resource_type = qrt.AndroidTvRt2842HikeenOtaImage
        default_repository_url = 'ssh://gerrit.yandex-team.ru/hikeen2842/manifest'
        yandex_prebuilt_repo_path = 'platform/kernel/android/pie/vendor/yandex/prebuilt'
        error_logo_path = None
        recovery_logo_path = None

        @property
        def rom_path(self):
            rom_dir_path = os.path.join(self.checkout_path, 'platform', 'image_file_creator')
            for node in os.listdir(rom_dir_path):
                if node.startswith('install_2842') and node.endswith('.img'):
                    return os.path.join(rom_dir_path, node)

        @property
        def ota_path(self):
            return os.path.join(self.checkout_path, 'platform/kernel/android/pie/signed-ota_update.zip')

        @property
        def _target_product_path(self):
            return os.path.join(
                self.checkout_path,
                'platform/kernel/android/pie/out/target/product/RealtekATV',
            )

        @property
        def vendor_config_path(self):
            return os.path.join(self._target_product_path, 'system/app/VendorConfig/VendorConfig.apk')

        @property
        def boot_logo_path(self):
            return os.path.join(self._target_product_path, 'bootfile.raw')

    def _build(self):
        os.chdir(os.path.join(self.checkout_path, 'platform'))

        make_cmd_list = [
            'make',
            'BRANCH_NAME=' + self.repository_tag,
            'PLATFORM_BOARD_NAME=' + self.config.platform_board,
            'DEVICE_KEY=' + self.device_key_path,
        ]
        if self.Parameters.build_type == build_types.ImageBuildtype.USER:
            make_cmd_list.append('all_user')
        else:
            make_cmd_list.append('all')

        shutil.copy(
            'hikeen/order/{build_product_target}/.old{build_product_target}config'.format(
                build_product_target=self.target_config['target'],
            ),
            '.config'
        )

        build_env = dict(os.environ)
        build_env.update({
            'BUILD_NUMBER': str(self.id),
        })

        self.run_command_with_log(make_cmd_list, 'build_image', env=build_env)
        os.chdir(str(self.path()))

    def fill_rom_dir(self, rom_dir_path):
        shutil.copy(self.config.rom_path, rom_dir_path)
