from sandbox import sdk2

from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.image_builds.base_image_build import buildroot as buildroot_base
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base


class QuasarBuildSaturnImage(buildroot_base.QuasarBaseBuildrootImageBuildTask):
    """
    Build buildroot-based image for quasar Saturn device and publish it as a resource.
    """

    BUILDROOT_CONFIG = 'yandex_saturn'
    BUILDROOT_CONFIG_RECOVERY = 'yandex_saturn_recovery'
    BUILDROOT_CONFIG_SL_LINUX = 'yandex_saturn_svc'
    BUILDROOT_CONFIG_HWTEST = 'yandex_saturn_hwtest'

    BUILD_TYPES_WITH_OTA_AND_DAEMONS = {
        # FIXME there is no quasar daemons yet, so commented
        # build_types.SaturnBuildrootImageBuildType.USER,
        # build_types.SaturnBuildrootImageBuildType.ENG
    }

    class Parameters(buildroot_base.QuasarBaseBuildrootImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.SATURN)
        quasar_daemons = param_creator.create_daemons_parameter()
        quasar_daemons_factory = param_creator.create_daemons_parameter(factory=True)
        ota_max_size = param_creator.create_ota_max_size_parameter()
        custom_manifest = param_creator.create_custom_manifest_parameter()

        # FIXME this is secret from micro
        ota_swupdate_key_secret = sdk2.parameters.String(
            'YAV secret with swupdate OTA rsa private key: priv.pem',
            default='ver-01enmdpnjy6zrgbzftnj8ej3fv')

        # FIXME this is secret from micro
        image_build_keys_secret = sdk2.parameters.String(
            'YAV secret with secureboot keys tarball, should have files: aml-user-key.sig, tdk_keys.tgz',
            default='sec-01em7bsjhjefxnah26tazv5vsk')

        with sdk2.parameters.String(
            'Build type', required=True,
            default=build_types.SaturnBuildrootImageBuildType.ENG,
        ) as build_type:
            for btype in list(build_types.SaturnBuildrootImageBuildType):
                build_type.values[btype] = btype

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexSmartDeviceLxcImage)

    class BuildConfig(buildroot_base.QuasarBaseBuildrootImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.SATURN

        @property
        def default_repository_tag(self):
            return 'saturn/master'

        @property
        def daemons_dest_path(self):
            return (self.checkout_path / 'vendor' / 'yandex' / 'board' / 'saturn' / 'overlay' / 'system' /
                    'vendor' / 'quasar')

        @property
        def factory_daemons_dest_path(self):
            return (self.checkout_path / 'vendor' / 'yandex' / 'board' / 'saturn' / 'overlay-recovery' / 'system' /
                    'vendor' / 'quasar')

        @property
        def buldroot_cache_platform(self):
            return self.default_buildroot_cache_platform

    @property
    def buildroot_config_name(self):
        if self.Parameters.buildroot_defconfig:
            return self.Parameters.buildroot_defconfig

        if self.Parameters.build_type == build_types.SaturnBuildrootImageBuildType.SL_LINUX:
            return self.BUILDROOT_CONFIG_SL_LINUX
        elif self.Parameters.build_type == build_types.SaturnBuildrootImageBuildType.HW_TEST:
            return self.BUILDROOT_CONFIG_HWTEST
        else:
            return self.BUILDROOT_CONFIG

    @property
    def buildroot_recovery_config_name(self):
        if self.Parameters.buildroot_recovery_defconfig:
            return self.Parameters.buildroot_recovery_defconfig

        return self.BUILDROOT_CONFIG_RECOVERY

    @property
    def with_quasar_daemons(self):
        return self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS

    def prepare_build_command(self):
        build_command = '. ./build/envsetup.sh'

        ta_keys_path = self.yav_export_tar(
            str(self.Parameters.image_build_keys_secret), 'tdk_keys.tgz'
        )

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            # build recovery for OTA before main build
            build_command += ' && lunch {} && m BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH={}'.format(
                self.buildroot_recovery_config_name, ta_keys_path)
            if self.Parameters.build_type in {
                build_types.SaturnBuildrootImageBuildType.USER
            }:
                build_command += ' BR2_TARGET_ADB_INITD_FILE=""'

        config = {
            'BR2_TARGET_ROOTFS_AMLOGIC_SIGN_KEY': self.yav_export_file(
                str(self.Parameters.image_build_keys_secret), 'aml-user-key.sig'
            ),
            'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH': ta_keys_path
        }

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            config['BR2_TARGET_ROOTFS_SWUPDATE_PRIV_KEY'] = self.yav_export_file(
                str(self.Parameters.ota_swupdate_key_secret), 'priv.pem'
            )
            recovery_path = str(self.config.checkout_path / 'out' / self.buildroot_recovery_config_name / 'images' /
                                'fastboot.img')
            config['BR2_TARGET_ROOTFS_AMLOGIC_RECOVERY'] = recovery_path
            if self.Parameters.ota_with_uboot:
                config['BR2_TARGET_ROOTFS_SWUPDATE_OTA_WITH_UBOOT'] = 'y'

        # silent console & rabbit hole setting
        if self.Parameters.build_type in {
                build_types.SaturnBuildrootImageBuildType.USER
        }:
            config['BR2_TARGET_ADB_INITD_FILE'] = '""'
            config['BR2_TARGET_AMLOGIC_UBOOT_SILENT_CONSOLE'] = 'y'
            config['BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY'] = self.yav_export_file(
                str(self.Parameters.image_build_keys_secret), 'rh_public.pem')

        build_command += ' && lunch {} && m {}'.format(
            self.buildroot_config_name,
            ' '.join('{}={}'.format(k, v) for k, v in config.items())
        )

        return build_command

    def publish_resources(self):
        version = self._determine_version() if self.with_quasar_daemons else None
        is_fastboot = self.Parameters.build_type in {build_types.SaturnBuildrootImageBuildType.SL_LINUX}
        self.publish_image(resource_type=qrt.QuasarSaturnImage,
                           secureboot=True,
                           fastboot=is_fastboot,
                           version=version)
        self.publish_image(resource_type=qrt.QuasarSaturnImage,
                           secureboot=False,
                           fastboot=is_fastboot,
                           version=version)

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            published_ota_resources = self.publish_ota_image(resource_type=qrt.QuasarSaturnOTAImage,
                                                             secureboot=True,
                                                             version=version)

            self.publish_ota_image(resource_type=qrt.QuasarSaturnOTAImage,
                                   secureboot=False,
                                   version=version)

            if self.Parameters.push_sensors_to_solomon:
                quasar_sizes = quasar_utils.get_quasar_sizes(self.config.daemons_dest_path)
                quasar_utils.push_sizes_to_solomon(
                    self.Parameters.quasar_daemons,
                    quasar_sizes,
                    published_ota_resources[qrt.QuasarSaturnOTAImage],
                )

            quasar_utils.check_ota_size(published_ota_resources[qrt.QuasarSaturnOTAImage],
                                        self.Parameters.ota_max_size)
