import logging
import shutil
import os

from sandbox import sdk2
from sandbox.sandboxsdk import paths as sandbox_paths
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.errors import TaskFailure

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.utils import vcs
from sandbox.projects.quasar.image_builds.base_image_build import linkplay as linkplay_base
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base


class LoggingLevel(quasar_utils.ListableEnum):
    NOT_SET = 'NOT_SET'
    DEBUG = 'DEBUG'
    INFO = 'INFO'
    WARNING = 'WARNING'
    ERROR = 'ERROR'
    CRITICAL = 'CRITICAL'


class QuasarBuildYandexminiImage(
        quasar_base.BaseQuasarImageBuildTask, quasar_utils.YAVExportMixin,
):
    """
    Build full image for quasar Station Mini device and publish it as a resource.
    """

    class Parameters(linkplay_base.QuasarLinkplayImageBuildTask.Parameters):
        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexminiLxcImage
        )
        kill_timeout = 60 * 60

        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXMINI)

        ota_max_size = param_creator.create_ota_max_size_parameter()
        quasar_daemons = param_creator.create_daemons_parameter()
        custom_manifest = param_creator.create_custom_manifest_parameter()

        prjs_branches = sdk2.parameters.Dict(
            "Branches in Bitbacket to build with",
            description="Key-value pairs of "
                        "local paths of projects relative to the repo's root "
                        "and branches"
        )

        with sdk2.parameters.String(
            'Build type', required=True,
            default=build_types.BuildrootImageBuildType.RELEASE,
        ) as build_type:
            for btype in list(build_types.BuildrootImageBuildType):
                build_type.values[btype] = btype

        sign = sdk2.parameters.Bool('Build signed image and OTA', default=True)

        with_mini_quasar = sdk2.parameters.Bool('Build with mini quasar', default=False)
        with with_mini_quasar.value[True]:
            mini_quasar = quasar_utils.LastResourceWithAttrs(
                'Mini quasar daemons',
                resource_type=qrt.QuasarMiniDaemons,
                attrs=dict(released=ctt.ReleaseStatus.TESTING),
            )

        with sdk2.parameters.String(
            'Logging level', required=False,
            default=LoggingLevel.DEBUG,
        ) as log_level:
            for lvl in LoggingLevel.all():
                log_level.values[lvl] = lvl

        sign_key_yav_version = sdk2.parameters.String(
            'YAV version for secret with key',
            default='ver-01dc0868j1s243ndhgqzc4ycmd'
        )
        sign_key_yav_key = 'yandex-station-mini-lplay-schematic'

    class Requirements(quasar_utils.YAVExportMixin.Requirements, sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.SSD  # repo is big
        disk_space = 16 * 1024  # 16 Gb

    class BuildConfig(quasar_base.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.YANDEXMINI

        @property
        def default_vcs(self):
            return vcs.VCS.REPO

        @property
        def default_repository_url(self):
            return 'ssh://bb.yandex-team.ru/quas/buildroot-manifest.git'

        @property
        def default_repository_tag(self):
            return 'yandexstation-mini'

        @property
        def base_dest_path(self):
            return self.checkout_path / 'buildroot' / 'board' / 'amlogic' / 'yandexstation_mini_lplay'

        @property
        def daemons_dest_path(self):
            return self.base_dest_path / 'rootfs' / 'system' / 'vendor' / 'quasar'

        @property
        def mini_quasar_dest_path(self):
            return self.base_dest_path / 'rootfs_mini_quasar' / 'system' / 'vendor' / 'quasar'

        @property
        def ramfslist_recovery(self):
            """ Text file with list of paths needed for recovery """
            return self.base_dest_path / 'ota' / 'swu' / 'ramfslist-recovery-need'

        @property
        def strip_tool_path(self):
            return self.checkout_path.joinpath(
                'toolchain',
                'gcc',
                'linux-x86',
                'arm',
                'gcc-linaro-6.3.1-2017.05-x86_64_arm-linux-gnueabihf',
                'bin',
                'arm-linux-gnueabihf-strip'
            )

        @property
        def release_build_type(self):
            return build_types.BuildrootImageBuildType.RELEASE

        @property
        def base_output_path(self):
            return self.checkout_path.joinpath(
                'output',
                'yandexstation-mini_lplay_32_{}'.format(self.task.Parameters.build_type),
                'images'
            )

        def get_build_artifacts(self, signed):
            image_name = 'aml_upgrade_package{}.img'.format('_enc' if signed else '')
            ota_name = 'packed_ota.data{}'.format('.signed' if signed else '')

            return {
                qrt.QuasarYandexminiImage: str(self.base_output_path / image_name),
                qrt.QuasarYandexminiOTAImage: str(self.base_output_path / ota_name),
            }

        def get_unpacked_ota_filename(self):
            unpacked_ota_name = 'yandex_io_ota/yandex_io_ota.swu'
            return str(self.base_output_path / unpacked_ota_name)

        @property
        def buildroot_cache_dest_path(self):
            return self.checkout_path / 'buildroot' / 'dl'

        @property
        def buldroot_cache_platform(self):
            return None

    @property
    def config(self):
        # type: () -> BuildConfig
        return self._config

    class RepoParameters(quasar_base.BaseQuasarImageBuildTask.RepoParameters):
        init_depth = None
        current_branch = False

    def repo_checkout(self):
        super(QuasarBuildYandexminiImage, self).repo_checkout()

        for prj, br, cmds in self.Parameters.prjs_branches:
            sandbox_process.run_process(
                [
                    'repo',
                    'forall',
                    prj,
                    '-c', '; '.join(
                        [
                            # TODO: Get remote branch and repo names from manifest file
                            'git fetch quasar refs/heads/{target_branch}'.format(target_branch=br),
                            'git branch --track {target_branch} quasar/{target_branch}'.format(target_branch=br),
                            'git checkout {target_branch}'.format(target_branch=br)
                        ]
                    )
                ],
                work_dir=self.checkout_path,
                log_prefix='repo_pull_branches'
            )

    def _find_mini_quasar_resource(self):
        """
        Find the resource with Mini Quasar
        """

        if self.Parameters.mini_quasar:
            return self.Parameters.mini_quasar

        # Get Mini Quasar from the task with the main Quasar.
        mini_quasar_daemons = sdk2.Resource.find(
            resource_type=qrt.QuasarMiniDaemons,
            task=self.Parameters.quasar_daemons.task,
        ).first()

        if mini_quasar_daemons is None:
            raise ValueError("Quasar daemons' parent task didn't generate resource with Mini Quasar")
        else:
            logging.info('Found resource %s with id %d', mini_quasar_daemons, mini_quasar_daemons.id)
            return mini_quasar_daemons

    def _fix_ramfs_list(self):
        ramfsed_paths = []

        ramfsed_quasar_prefix_path = sdk2.path.Path('./system/vendor/quasar')

        with open(str(self.config.ramfslist_recovery), 'r+') as ramfs_list_file:
            for line in ramfs_list_file:
                ramfsed_path = sdk2.path.Path(line.rstrip())

                # If `relative_to` method doesn't raise an exception, `ramfsed_path`
                # is not relative to Mini Quasar path (`ramfsed_quasar_prefix_path`).
                # And we can add non-relative path to list of path.
                try:
                    ramfsed_path.relative_to(ramfsed_quasar_prefix_path)
                except ValueError:
                    ramfsed_paths.append(ramfsed_path)

            mini_quasar_path = self.config.mini_quasar_dest_path
            for root, dirs, files in os.walk(mini_quasar_path):
                # Make Mini Quasar paths with format supported by cpio.
                root = ramfsed_quasar_prefix_path / sdk2.path.Path(root).relative_to(mini_quasar_path)
                ramfsed_paths.append(root)

                for f in files:
                    ramfsed_paths.append(root / f)

            logger = quasar_utils.create_task_logger(self, 'ramfs_list.out.log', 'ramfs_list_fixing')

            ramfs_list_file.seek(0)
            ramfs_list_file.truncate(0)
            for ramfsed_path in sorted(ramfsed_paths, key=lambda p: str(p)):
                ramfs_list_entry = './' + str(ramfsed_path) + '\n'
                ramfs_list_file.write(ramfs_list_entry)

                logger.debug(ramfs_list_entry)

    def _place_mini_quasar(self):
        """
        Find and place Mini Quasar for recovery image.

        First, find the resource with Mini Quasar with requested attributes.
        If not given in parameters, task will try to get Mini Quasar from
        the same task, that created main daemons.

        Second, we must fix ``ramfslist-recovery-need`` file to make in containable our
        Mini Quasar.
        """
        resource = self._find_mini_quasar_resource()

        dest_path = self.config.mini_quasar_dest_path
        download_path = str(sdk2.ResourceData(resource).path)

        logging.info('Downloaded resource %s to %s', resource, download_path)

        sandbox_paths.remove_path(dest_path)
        sandbox_paths.copy_path(download_path, dest_path, symlinks=True)
        self._add_write_permissions(dest_path)

        logging.debug('Copied Mini Quasar from %s to %s', download_path, dest_path)

        self._fix_ramfs_list()

    def _run_build_script(self, build_arguments, env, log_prefix):
        sandbox_process.run_process(
            [
                'bash',
                'build_all_ci.sh',
            ] + build_arguments,
            log_prefix='build_all_ci.sh',
            environment=env,
            work_dir=self.checkout_path,
        )

    def _on_execute(self):
        os.environ.update({
            'SANDBOXED': 'true'
        })

        self._checkout()
        self._place_package_cache()

        self._place_daemons()
        self._strip_binaries()
        quasar_sizes = quasar_utils.get_quasar_sizes(self.config.daemons_dest_path)

        # Place version everywhere in quasar code.
        version = self._determine_version(self.config.checkout_path / 'build-yio' / 'VERSION')
        self._place_version(version)

        if self.Parameters.with_mini_quasar:
            self._place_mini_quasar()
            self._strip_binaries(self.config.mini_quasar_dest_path)
            self._place_version(version, self.config.mini_quasar_dest_path)

        signing_key_name = self.Parameters.sign_key_yav_key
        signing_key_path = self.yav_export(self.Parameters.sign_key_yav_version, signing_key_name)

        build_arguments = [
            '--flavour', 'lplay',
            self.Parameters.build_type,
            'signed' if self.Parameters.sign else 'unsigned',
        ]
        build_environment = {
            '_LOG_LEVEL': self.Parameters.log_level,
            'YANDEX_STATION_MINI_SIGN_KEY_PATH': signing_key_path,
            'YANDEX_STATION_MINI_SIGN_KEY_FILENAME': signing_key_name,
        }
        self._build(build_arguments, build_environment)
        shutil.rmtree(signing_key_path)

        resource_attrs = dict(
            buildtype=self.Parameters.build_type,
            version=version,
        )

        unsigned_resource_attrs = dict(resource_attrs)
        unsigned_resource_attrs['signed'] = False

        self._publish(
            self.config.get_build_artifacts(signed=False),
            unsigned_resource_attrs,
        )

        if self.Parameters.sign:
            signed_resource_attrs = dict(resource_attrs)
            signed_resource_attrs['signed'] = True

            signed_resources = self._publish(
                self.config.get_build_artifacts(signed=True),
                signed_resource_attrs,
            )

            if self.Parameters.push_sensors_to_solomon:
                quasar_utils.push_sizes_to_solomon(
                    self.Parameters.quasar_daemons,
                    quasar_sizes,
                    signed_resources[qrt.QuasarYandexminiOTAImage],
                )

            self.check_unpacked_ota_size(self.config.get_unpacked_ota_filename(), self.Parameters.ota_max_size)

# this function need only for mini1
# don't need to integrate it to quasar_utils
    def check_unpacked_ota_size(self, ota_file, size_limit_mb):
        if size_limit_mb is None:
            logging.info("Not checking size limit info for resource because size limit is None")
            return

        size_limit = size_limit_mb * 1024 * 1024

        actual_size = os.path.getsize(ota_file)

        if actual_size > size_limit:
            raise TaskFailure('OTA too big: {} > {} limit! {} bytes above limit.'.format(
                actual_size,
                size_limit,
                actual_size - size_limit
            ))
