import os

from sandbox import sdk2

from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base
from sandbox.projects.quasar.image_builds.base_image_build import buildroot as buildroot_base


class QuasarBuildYandexmini2Image(buildroot_base.QuasarBaseBuildrootImageBuildTask):
    """
    Build buildroot-based image for quasar Station Mini 2 device and publish it as a resource.
    """

    BUILDROOT_CONFIG = 'yandex_mini2'
    BUILDROOT_CONFIG_RECOVERY = 'yandex_mini2_recovery'
    BUILDROOT_CONFIG_HWTEST = 'yandex_mini2_hwtest'
    BUILDROOT_CONFIG_SL_LINUX = 'yandex_mini2_svc'

    BUILD_TYPES_WITH_OTA_AND_DAEMONS = {
        build_types.Yandexmini2BuildrootImageBuildType.FACTORY,
        build_types.Yandexmini2BuildrootImageBuildType.USER,
        build_types.Yandexmini2BuildrootImageBuildType.ENG
    }

    class Parameters(buildroot_base.QuasarBaseBuildrootImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXMINI_2)
        quasar_daemons = param_creator.create_daemons_parameter()
        quasar_daemons_factory = param_creator.create_daemons_parameter(factory=True)
        ota_max_size = param_creator.create_ota_max_size_parameter()
        custom_manifest = param_creator.create_custom_manifest_parameter()

        ota_swupdate_key_secret = sdk2.parameters.String(
            'YAV secret with swupdate OTA rsa private key: priv.pem',
            default='ver-01ej5t3ykk6ry1b1w0dwqfywdx')

        image_build_keys_secret = sdk2.parameters.String(
            'YAV secret with secureboot keys tarball, should have files: aml-user-key.sig, tdk_keys.tgz',
            default='sec-01ejr60q44g8dm2xnqj15g40c6')

        with sdk2.parameters.String(
            'Build type', required=True,
            default=build_types.Yandexmini2BuildrootImageBuildType.ENG,
        ) as build_type:
            for btype in list(build_types.Yandexmini2BuildrootImageBuildType):
                build_type.values[btype] = btype

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexmini2LxcImage)

    class BuildConfig(buildroot_base.QuasarBaseBuildrootImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.YANDEXMINI_2

        @property
        def default_repository_tag(self):
            return 'yandexmini_2/master'

        @property
        def daemons_dest_path(self):
            return (self.checkout_path / 'vendor' / 'yandex' / 'board' / 'mini2' / 'overlay' / 'system' /
                    'vendor' / 'quasar')

        @property
        def factory_daemons_dest_path(self):
            return (self.checkout_path / 'vendor' / 'yandex' / 'board' / 'mini2' / 'overlay-recovery' / 'system' /
                    'vendor' / 'quasar')

        @property
        def buldroot_cache_platform(self):
            return self.default_buildroot_cache_platform

    @property
    def buildroot_config_name(self):
        if self.Parameters.buildroot_defconfig:
            return self.Parameters.buildroot_defconfig

        if self.Parameters.build_type == build_types.Yandexmini2BuildrootImageBuildType.SL_LINUX:
            return self.BUILDROOT_CONFIG_SL_LINUX
        elif self.Parameters.build_type == build_types.Yandexmini2BuildrootImageBuildType.HW_TEST:
            return self.BUILDROOT_CONFIG_HWTEST
        else:
            return self.BUILDROOT_CONFIG

    @property
    def buildroot_recovery_config_name(self):
        if self.Parameters.buildroot_recovery_defconfig:
            return self.Parameters.buildroot_recovery_defconfig

        return self.BUILDROOT_CONFIG_RECOVERY

    @property
    def with_quasar_daemons(self):
        return self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS

    def prepare_build_command(self):
        build_command = '. ./build/envsetup.sh'

        ta_keys_path = self.yav_export_tar(
            str(self.Parameters.image_build_keys_secret), 'tdk_keys.tgz'
        )

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            # build recovery for OTA before main build
            build_command += ' && lunch {} && m BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH={}'.format(
                self.buildroot_recovery_config_name, ta_keys_path)

        config = {
            'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'y',
            'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'y',
            'BR2_TARGET_ROOTFS_AMLOGIC_SIGN_KEY': self.yav_export_file(
                str(self.Parameters.image_build_keys_secret), 'aml-user-key.sig'
            ),
            'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH': ta_keys_path
        }

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            config['BR2_TARGET_ROOTFS_SWUPDATE_PRIV_KEY'] = self.yav_export_file(
                str(self.Parameters.ota_swupdate_key_secret), 'priv.pem'
            )
            recovery_path = str(self.config.checkout_path / 'out' / self.buildroot_recovery_config_name / 'images' /
                                'fastboot.img')
            config['BR2_TARGET_ROOTFS_AMLOGIC_RECOVERY'] = recovery_path
            if self.Parameters.ota_with_uboot:
                config['BR2_TARGET_ROOTFS_SWUPDATE_OTA_WITH_UBOOT'] = 'y'

        # factory mode setting
        if self.Parameters.build_type in {
            build_types.Yandexmini2BuildrootImageBuildType.FACTORY
        }:
            config['BR2_TARGET_AMLOGIC_UBOOT_WITH_FACTORY_MODE'] = 1
        elif self.Parameters.build_type in {
            build_types.Yandexmini2BuildrootImageBuildType.USER,
            build_types.Yandexmini2BuildrootImageBuildType.ENG
        }:
            config['BR2_TARGET_AMLOGIC_UBOOT_WITH_FACTORY_MODE'] = 0
        elif self.Parameters.build_type in {
            build_types.Yandexmini2BuildrootImageBuildType.HW_TEST
        }:
            config['BR2_TARGET_AMLOGIC_UBOOT_WITH_FACTORY_MODE'] = 2

        # silent console & rabbit hole setting
        if self.Parameters.build_type in {
            build_types.Yandexmini2BuildrootImageBuildType.FACTORY,
            build_types.Yandexmini2BuildrootImageBuildType.USER
        }:
            config['BR2_TARGET_AMLOGIC_UBOOT_SILENT_CONSOLE'] = 'y'
            config['BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY'] = self.yav_export_file(
                str(self.Parameters.image_build_keys_secret), 'rh_public.pem')

        build_command += ' && lunch {} && m {}'.format(
            self.buildroot_config_name,
            ' '.join('{}={}'.format(k, v) for k, v in config.items())
        )

        return build_command

    def publish_resources(self):
        version = self._determine_version() if self.with_quasar_daemons else None
        is_fastboot = self.Parameters.build_type in {build_types.Yandexmini2BuildrootImageBuildType.SL_LINUX}
        self.publish_image(resource_type=qrt.QuasarYandexmini2Image,
                           secureboot=True,
                           fastboot=is_fastboot,
                           version=version)
        self.publish_image(resource_type=qrt.QuasarYandexmini2Image,
                           secureboot=False,
                           fastboot=is_fastboot,
                           version=version)

        if self.Parameters.build_type == build_types.Yandexmini2BuildrootImageBuildType.FACTORY:
            efuse_path = os.path.join('out', self.buildroot_config_name, 'build', 'amlogic-uboot-local', 'build',
                                      'u-boot.bin.encrypt.efuse')
            self._publish(resources={qrt.QuasarYandexmini2Efuse: efuse_path})

        if self.Parameters.build_type in self.BUILD_TYPES_WITH_OTA_AND_DAEMONS:
            ota_image_path = os.path.join('out', self.buildroot_config_name, 'images', 'yandex_io_ota.swu')
            published_ota_resources = self._publish(
                resources={
                    qrt.QuasarYandexmini2OTAImage: ota_image_path
                },
                resources_attrs=dict(
                    version=version,
                    signed=True,
                    buildtype=self.Parameters.build_type
                )
            )

            if self.Parameters.push_sensors_to_solomon:
                quasar_sizes = quasar_utils.get_quasar_sizes(self.config.daemons_dest_path)
                quasar_utils.push_sizes_to_solomon(
                    self.Parameters.quasar_daemons,
                    quasar_sizes,
                    published_ota_resources[qrt.QuasarYandexmini2OTAImage],
                )

            quasar_utils.check_ota_size(published_ota_resources[qrt.QuasarYandexmini2OTAImage],
                                        self.Parameters.ota_max_size)
