import os
import logging
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import client as ctc

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar.resource_types import tv as tvrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.utils import vcs
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base


class QuasarBuildYandexmodule2LinuxImage(
        quasar_utils.YAVExportMixin,
        quasar_base.BaseQuasarImageBuildTask
):
    """
    Build buildroot-based Linux image for yandexmodule_2 sideload in factory
    mode and publish it as a resource.
    """

    BUILDROOT_CONFIG = 'yandex_module2_svc'

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXMODULE_2)

        kill_timeout = 1 * 60 * 60  # 1h

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")
        custom_manifest = param_creator.create_custom_manifest_parameter()

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool(
                'Suspend task for debug purposes just before actual build starts',
                default=False)

        secureboot = sdk2.parameters.Bool('Build secureboot image', default=True)
        secureboot_keys_secret = sdk2.parameters.String(
            'YAV secret should have file aml-user-key.sig',
            default='ver-01f00r489nx20dk480gjhfha7c')
        tdk_keys_secret = sdk2.parameters.String(
            'YAV secret should have file tdk_keys.tgz',
            default='ver-01exp6j4ydksk8c7nw4x2fe3v7')
        provision_protection_key_secret = sdk2.parameters.String(
            'YAV secret with provision common protection key',
            default='ver-01ee4sard725md74n1qjw3zyxg')

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=tvrt.QuasarYandexmodule2LxcImage)

        build_type = None

    class Requirements(quasar_utils.YAVExportMixin.Requirements,
                       quasar_base.BaseQuasarImageBuildTask.Requirements):
        client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.SSD
        disk_space = 30 * 1024  # 30 Gb

    class BuildConfig(quasar_base.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.YANDEXMODULE_2

        @property
        def default_vcs(self):
            return vcs.VCS.REPO

        @property
        def default_repository_url(self):
            return 'ssh://gerrit.yandex-team.ru/yandex-io/buildroot-manifest'

        @property
        def default_repository_tag(self):
            return 'yandexmodule_2/master'

        @property
        def buildroot_cache_dest_path(self):
            return self.checkout_path / 'dist'

        def daemons_dest_path(self):
            return None

        @property
        def strip_tool_path(self):
            return None

        @property
        def buldroot_cache_platform(self):
            return self.default_buildroot_cache_platform

        def load_package_config(self):
            self.vcs = self.default_vcs
            self.repository_url = self.default_repository_url
            self.repository_tag = self.default_repository_tag

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def _on_execute(self):
        self._checkout()
        self._place_package_cache()

        secureboot = bool(self.Parameters.secureboot)

        if secureboot:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'y',
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH': self.yav_export_tar(
                    str(self.Parameters.tdk_keys_secret), 'tdk_keys.tgz'),
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN_KEY': self.yav_export_file(
                    str(self.Parameters.secureboot_keys_secret),
                    'aml-user-key.sig'),
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'y'}
        else:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'n',
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'n'}

        config['BR2_PACKAGE_PROVISION_PROTECTION_KEY'] = self.yav_export_file(
            str(self.Parameters.provision_protection_key_secret), 'pcpk.bin')

        build_command = '. ./build/envsetup.sh && lunch {} && m {}'.format(
            self.BUILDROOT_CONFIG,
            ' '.join('{}={}'.format(k, v) for k, v in config.items()),
        )

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Execute command %s', build_command)
            self.suspend()

        sandbox_process.run_process(
            ['bash', '-c', build_command],
            log_prefix=self.BUILDROOT_CONFIG,
            work_dir=self.checkout_path,
        )

        image_path = os.path.join('out', self.BUILDROOT_CONFIG, 'images',
                                  'fastboot.img')
        image_enc_path = os.path.join('out', self.BUILDROOT_CONFIG, 'images',
                                      'fastboot-signed.img')

        if secureboot:
            self._publish(
                resources={
                    tvrt.QuasarYandexmodule2LinuxImage: image_enc_path},
                resources_attrs=dict(
                    secureboot=True,
                ))
        self._publish(
            resources={
                tvrt.QuasarYandexmodule2LinuxImage: image_path},
            resources_attrs=dict(
                secureboot=False,
            ))

    def yav_export_file(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of file to export
        :return str: path to exported file
        """
        return os.path.join(self.yav_export(secret_or_version, key), key)

    def yav_export_tar(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of tar archive to export
        :return str: path to exported dir
        """
        export_path = self.yav_export(secret_or_version, key)
        tar_path = os.path.join(export_path, key)
        extract_dir = tempfile.mkdtemp(prefix=key, dir=export_path)
        with tarfile.open(tar_path) as tar:
            tar.extractall(extract_dir)
        return extract_dir
