import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import paths as sandbox_paths
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.utils import vcs
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base


class QuasarBuildYandexstationImage(quasar_utils.SignerMixin, quasar_base.BaseQuasarImageBuildTask):
    """
    Build full image for quasar device and publish it as a resource
    Optionally, sign it via Signer
    """

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXSTATION)

        kill_timeout = 5 * 3600  # can take long, duh

        ota_max_size = param_creator.create_ota_max_size_parameter()
        sign = sdk2.parameters.Bool('Build signed image and OTA', default=False)
        quasar_app = quasar_utils.LastStableResource('Quasar App .apk', resource_type=qrt.QuasarApp)
        quasar_daemons = param_creator.create_daemons_parameter()

        perf = sdk2.parameters.Bool('Build performance farm image', default=False)
        quasar_stub = quasar_utils.LastStableResource('Quasar Stub .apk', resource_type=qrt.QuasarStub)

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexstationLxcImage)

        with quasar_base.BaseQuasarImageBuildTask.Parameters.internal_block() as internal_block:
            c_cache_tarball = quasar_utils.LastResource('Quasar android ccache dir tarball', resource_type=qrt.QuasarImageAndroidCcacheTarball)

    class Requirements(quasar_base.BaseQuasarImageBuildTask.Requirements):
        dns = ctm.DnsType.DNS64  # for external interactions
        privileged = True  # for _place_version method
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.SSD  # repo is big
        disk_space = 150 * 1024  # 150 Gb

        environments = [
        ] + quasar_utils.SignerMixin.environments

    class BuildConfig(quasar_base.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.YANDEXSTATION

        @property
        def default_vcs(self):
            return vcs.VCS.GIT

        @property
        def default_repository_url(self):
            return 'ssh://bb.yandex-team.ru/quas/r18stationandroid.git'

        @property
        def base_dest_path(self):
            return self.checkout_path / 'R18_android' / 'android' / 'device' / 'softwinner' / 'tulip-d1'

        @property
        def daemons_dest_path(self):
            return self.base_dest_path / 'quasar'

        @property
        def strip_tool_path(self):
            return self.checkout_path.joinpath(
                'R18_android',
                'android',
                'prebuilts',
                'gcc',
                'linux-x86',
                'aarch64',
                'aarch64-linux-android-4.9',
                'aarch64-linux-android',
                'bin',
                'strip'
            )

        @property
        def apk_dest_path(self):
            return self.base_dest_path / 'packages' / 'Quasar-Webmusic' / 'webmusic.apk'

        @property
        def stub_dest_path(self):
            return self.base_dest_path / 'packages' / 'Quasar-Stub' / 'stub.apk'

    class GitParameters(vcs.GitCheckoutMixin.GitParameters):
        pull_lfs = True

    CCACHE_DIR = '__ccache'  # we always use this as CCACHE_DIR
    CCACHE_SIZE = '20G'

    def _place_daemons(self):
        super(QuasarBuildYandexstationImage, self)._place_daemons()

        if self.Parameters.perf:
            source_run_sh = str(self.config.daemons_dest_path / 'run_perf.sh')
            dest_run_sh = str(self.config.daemons_dest_path / 'run.sh')
            sandbox_paths.remove_path(dest_run_sh)
            sandbox_paths.copy_path(source_run_sh, dest_run_sh, symlinks=True)

    def _place_resources(self):
        super(QuasarBuildYandexstationImage, self)._place_resources()

        # location is relatively to the repo root
        resource_to_location = {
            self.Parameters.quasar_app: str(self.config.apk_dest_path),
        }
        if self.Parameters.perf:
            resource_to_location[self.Parameters.quasar_stub] = str(self.config.stub_dest_path)

        self._place_resources_to_location(resource_to_location)

    def _ccache_command(self, args, log_prefix='ccache'):
        # TODO: we are using this ccache for now, may need tweaking
        ccache = 'R18_android/android/prebuilts/misc/linux-x86/ccache/ccache'

        env = os.environ.copy()
        env['CCACHE_DIR'] = str(self.path(self.CCACHE_DIR))

        return sandbox_process.run_process(
            [str(self.path(self.checkout_path, ccache))] + list(args),
            environment=env,
            log_prefix=log_prefix)

    def _ccache_stats(self):
        """
        Calls ccache prebuilt binary to show stats
        """
        return self._ccache_command(['-s'], log_prefix='ccache_stats')

    def _prepare_ccache(self):
        self.untarball_dir_res(self.Parameters.c_cache_tarball, self.path(self.CCACHE_DIR))

        self._ccache_command(['-M', self.CCACHE_SIZE], log_prefix='ccache_set_max_size')

        self._ccache_stats()

    def _publish_ccache(self):
        self._ccache_stats()

        self.tarball_publish_dir(qrt.QuasarImageAndroidCcacheTarball, self.path(self.CCACHE_DIR), comment='ccache for android')

    def sign_images(self, target_files_resource):
        """
        See https://st.yandex-team.ru/SIGNER-173 -- signed archive always contains two files with predefined names

        :param target_files_resource: to sign
        :returns: pair `(signed_target_files_path, signed_ota_path)`
        """
        signed_bundle = self.sign(sdk2.ResourceData(target_files_resource).path, quasar_utils.SignerMixin.Certs.TARGET_FILES)

        logging.info("Signed target files to %s" % (signed_bundle))

        extracted_path = str(self.path('_signed_target_files'))
        os.makedirs(extracted_path)

        sandbox_process.run_process(
            ['unzip', signed_bundle, '-d', extracted_path],
            log_prefix='extract_bundle',
        )

        return [os.path.join(extracted_path, element) for element in ('target_files.zip', 'ota.zip')]

    def _on_execute(self):
        self._checkout()

        self._place_resources()

        # TODO: do it only for USER versions of image
        if self.Parameters.quasar_daemons.is_before(release=28, trunk_revision=7135281):
            # Daemons must be stripped if they were built from trunk revision prior to 7135281
            # or from any release branch prior to 28 due to changes in stripping mechanism.
            # See: https://st.yandex-team.ru/SK-4401
            self._strip_binaries()

        quasar_sizes = quasar_utils.get_quasar_sizes(self.config.daemons_dest_path)
        apk_size = quasar_utils.get_disk_size(self.config.apk_dest_path)

        version = self._determine_version()
        self._place_version(version)

        self._prepare_ccache()

        extra_env = {
            # use bundled android ccache
            # see https://source.android.com/setup/initializing#setting-up-ccache
            'USE_CCACHE': '1',
            # use tina linux's ccache build option
            'CONFIG_CCACHE': '1',
            'CCACHE_DIR': str(self.path(self.CCACHE_DIR)),
        }

        if self.Parameters.perf:
            extra_env['BUILD_PERF_VARIANT'] = 'true'

        self._build(['kernel', self.Parameters.build_type])

        self._build(
            ['linux'],
            # FIXME: linux wont build with ccache now for some reaseon
            # environment=extra_env
        )

        self._publish(resources={qrt.QuasarLinuxImage: 'r18-linux/out/tulip-d1/tina_tulip-d1_uart0.img'})

        self._build(['android', self.Parameters.build_type], environment=extra_env)

        unsigned_resource_attrs = dict(
            buildtype=self.Parameters.build_type,
            version=version,
            signed=False,
        )

        unsigned_artifacts = self._publish(
            resources={
                qrt.QuasarYandexstationImage: 'R18_android/lichee/tools/pack/sun50iw1p1_android_d1_uart0.img',
                # file name is really different each time
                qrt.QuasarYandexstationTargetFiles: 'R18_android/android/out/target/product/tulip-d1/obj/PACKAGING/target_files_intermediates/tulip_d1-target_files-*.zip',
                qrt.QuasarYandexstationOTAImage: 'R18_android/android/out/target/product/tulip-d1/tulip_d1-ota-*.zip'},
            resources_attrs=unsigned_resource_attrs
        )

        if self.Parameters.sign:
            signed_targetfiles, signed_ota = self.sign_images(unsigned_artifacts[qrt.QuasarYandexstationTargetFiles])

            self._build(['signed_image', self.Parameters.build_type, signed_targetfiles], environment=extra_env)

            signed_resource_attrs = dict(
                buildtype=self.Parameters.build_type,
                version=version,
                signed=True,
            )

            signed_artifacts = self._publish(
                resources={
                    qrt.QuasarYandexstationImage: 'R18_android/lichee/tools/pack/sun50iw1p1_android_d1_uart0.img',
                    qrt.QuasarYandexstationOTAImage: signed_ota,  # take ota as was signed in signer
                },
                resources_attrs=signed_resource_attrs,
            )

            if self.Parameters.push_sensors_to_solomon and self.Parameters.build_type == build_types.ImageBuildtype.USER:
                quasar_utils.push_sizes_to_solomon(
                    self.Parameters.quasar_daemons,
                    quasar_sizes,
                    signed_artifacts[qrt.QuasarYandexstationOTAImage],
                    apk_size=apk_size,
                )

        self._ccache_stats()

        self._publish_ccache()

        quasar_utils.check_ota_size(unsigned_artifacts[qrt.QuasarYandexstationOTAImage], self.Parameters.ota_max_size)
        if self.Parameters.sign:
            quasar_utils.check_ota_size(signed_artifacts[qrt.QuasarYandexstationOTAImage], self.Parameters.ota_max_size)
