import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import process as sandbox_process

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar.image_builds.yandexstation2 import linux_image


class QuasarBuildYandexstation2BootloaderImage(
        linux_image.QuasarBuildYandexstation2LinuxImage):
    """
    Build buildroot-based Linux image and U-Boot image for yandexstation_2
    device flashing and publish it as a resource.
    """

    BUILDROOT_CONFIG = 'yandex_station2_installer'

    class Parameters(linux_image.QuasarBuildYandexstation2LinuxImage.Parameters):
        rabbit_hole_key_secret = sdk2.parameters.String(
            'YAV secret with rabbit_hole rsa private key: rh_key.pem',
            default='ver-01ee2njeqfzasqzc2y1jtn3kkv')
        avb_key_secret = sdk2.parameters.String(
            'YAV secret with avb rsa private key: avb_rsa2048.pem',
            default='ver-01eectr8kn0ps2xdprvpz6aj9t')
        provision_protection_key_secret = None
        base_station_2_image = quasar_utils.LastResourceWithAttrs(
            'Station2 image for which the bootloader is building for',
            resource_type=qrt.QuasarYandexstation2Image,
            attrs=dict(factory=True, buildtype="user"))

    def _on_execute(self):
        self._checkout()

        secureboot = bool(self.Parameters.secureboot)

        if secureboot:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'y',
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH': self.yav_export_tar(
                    str(self.Parameters.secureboot_keys_secret), 'tdk_keys.tgz'),
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN_KEY': self.yav_export_file(
                    str(self.Parameters.secureboot_keys_secret),
                    'aml-user-key.sig'),
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'y',
                'BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY': self.yav_export_file(
                    str(self.Parameters.rabbit_hole_key_secret), 'rh_key.pem'),
                'BR2_PACKAGE_AVB_SUPPORT_AVB_KEY_PATH': self.yav_export_file(
                    str(self.Parameters.avb_key_secret), 'avb_rsa2048.pem')}
        else:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'n',
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'n',
                'BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY': '',
                'BR2_PACKAGE_AVB_SUPPORT': 'n'}

        version = self.Parameters.base_station_2_image.version + ".bootloader"
        config['BR2_TARGET_AMLOGIC_UBOOT_BUILD_VERSION'] = version

        build_command = '. ./build/envsetup.sh && lunch {} && m {}'.format(
            self.BUILDROOT_CONFIG,
            ' '.join('{}={}'.format(k, v) for k, v in config.items()),
        )

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Execute command %s', build_command)
            self.suspend()

        sandbox_process.run_process(
            ['bash', '-c', build_command],
            log_prefix=self.BUILDROOT_CONFIG,
            work_dir=self.checkout_path,
        )

        fastboot_image_path = os.path.join('out', self.BUILDROOT_CONFIG,
                                           'images', 'fastboot.img')
        fastboot_image_enc_path = os.path.join('out', self.BUILDROOT_CONFIG,
                                               'images', 'fastboot-signed.img')
        install_image_path = os.path.join('out', self.BUILDROOT_CONFIG,
                                          'images', 'uboot-installer.img')
        install_image_enc_path = os.path.join('out', self.BUILDROOT_CONFIG,
                                              'images', 'uboot-installer-signed.img')

        if secureboot:
            self._publish(
                resources={
                    qrt.QuasarYandexstation2LinuxImage: fastboot_image_enc_path,
                    qrt.QuasarYandexstation2BootloaderImage: install_image_enc_path},
                resources_attrs=dict(
                    secureboot=True,
                    version=version,
                ))
        self._publish(
            resources={
                qrt.QuasarYandexstation2LinuxImage: fastboot_image_path,
                qrt.QuasarYandexstation2BootloaderImage: install_image_path},
            resources_attrs=dict(
                secureboot=False,
                version=version,
            ))

        self._publish_repo_manifest()
