import os
import logging
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import client as ctc
from sandbox.projects.quasar.utils import vcs
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base
from sandbox.projects.quasar import (
    utils as quasar_utils,
    resource_types as qrt,
    platform,
)


class QuasarBuildYandexstation2FactoryImages(quasar_utils.YAVExportMixin,
                                             quasar_base.BaseQuasarImageBuildTask):
    """
    Build buildroot-based Linux image and U-Boot for yandexstation_2 JIG/FP and publish it as a resource.
    """

    BUILDROOT_CONFIG_JIG = 'yandex_station2_jig'
    BUILDROOT_CONFIG_SVC = 'yandex_station2_svc'
    BUILDROOT_CONFIG_INS = 'yandex_station2_installer'  # for flash point

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXSTATION_2)

        kill_timeout = 3 * 60 * 60  # 3h

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")
        custom_manifest = param_creator.create_custom_manifest_parameter()

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool(
                'Suspend task for debug purposes just before actual build starts',
                default=False)

        build_fp_img = sdk2.parameters.Bool('Build Flash Point images', default=True)
        build_jig_img = sdk2.parameters.Bool('Build JIG images', default=True)
        build_sl_img = sdk2.parameters.Bool('Build SL linux', default=True)
        secureboot = sdk2.parameters.Bool('Build secureboot image', default=True)

        secureboot_keys_secret = sdk2.parameters.String(
            'YAV secret with secureboot keys tarball, should have files: aml-user-key.sig, tdk_keys.tgz',
            default='ver-01e9qdk5tnqvzehjkw6p4xm1h9')

        rabbit_hole_key_secret = sdk2.parameters.String(
            'YAV secret with rabbit_hole rsa private key: rh_key.pem',
            default='ver-01ee2njeqfzasqzc2y1jtn3kkv')
        avb_key_secret = sdk2.parameters.String(
            'YAV secret with avb rsa private key: avb_rsa2048.pem',
            default='ver-01eectr8kn0ps2xdprvpz6aj9t')

        provision_protection_key_secret = sdk2.parameters.String(
            'YAV secret with provision common protection key',
            default='ver-01ee4sard725md74n1qjw3zyxg')

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexstation2LxcImage)

        build_type = None

    class Requirements(quasar_utils.YAVExportMixin.Requirements,
                       quasar_base.BaseQuasarImageBuildTask.Requirements):
        client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.SSD
        disk_space = 60 * 1024  # 60 Gb

    class BuildConfig(quasar_base.BaseQuasarImageBuildTask.BuildConfig):
        @property
        def platform(self):
            return platform.Platform.YANDEXSTATION_2

        @property
        def default_vcs(self):
            return vcs.VCS.REPO

        @property
        def default_repository_url(self):
            return 'ssh://gerrit.yandex-team.ru/yandex-io/buildroot-manifest'

        @property
        def default_repository_tag(self):
            return 'master'

        @property
        def buildroot_cache_dest_path(self):
            return self.checkout_path / 'dist'

        def daemons_dest_path(self):
            return None

        @property
        def strip_tool_path(self):
            return None

        @property
        def buldroot_cache_platform(self):
            return self.default_buildroot_cache_platform

        def load_package_config(self):
            self.vcs = self.default_vcs
            self.repository_url = self.default_repository_url
            self.repository_tag = self.default_repository_tag

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def publish_svc(self, secureboot):
        fastboot_img = 'fastboot-signed.img' if secureboot else 'fastboot.img'
        fastboot_img_path = os.path.join('out', self.BUILDROOT_CONFIG_SVC, 'images', fastboot_img)

        self._publish(
            resources={qrt.QuasarYandexstation2LinuxImage: fastboot_img_path},
            resources_attrs=dict(secureboot=secureboot, )
        )

    def run_build(self, build_command):
        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Execute command %s', build_command)
            self.suspend()

        sandbox_process.run_process(
            ['bash', '-c', build_command],
            log_prefix=self.BUILDROOT_CONFIG_INS,
            work_dir=self.checkout_path,
        )

    def publish_fp(self, secureboot):
        fastboot_img = 'fastboot-signed.img' if secureboot else 'fastboot.img'
        fastboot_img_path = os.path.join('out', self.BUILDROOT_CONFIG_INS, 'images', fastboot_img)
        uboot_img = 'uboot-installer-signed.img' if secureboot else 'uboot-installer.img'
        uboot_img_path = os.path.join('out', self.BUILDROOT_CONFIG_INS, 'images', uboot_img)

        self._publish(
            resources={
                qrt.QuasarYandexstation2LinuxImageFP: fastboot_img_path,
                qrt.QuasarYandexstation2BootloaderImageFP: uboot_img_path},
            resources_attrs=dict(secureboot=secureboot)
        )

    def publish_jig(self, secureboot):
        fastboot_img = 'fastboot-signed.img' if secureboot else 'fastboot.img'
        fastboot_img_path = os.path.join('out', self.BUILDROOT_CONFIG_JIG, 'images', fastboot_img)
        uboot = 'u-boot.bin.encrypt' if secureboot else 'u-boot.bin'
        uboot_path = os.path.join('out', self.BUILDROOT_CONFIG_JIG, 'images', uboot)

        self._publish(
            resources={
                qrt.QuasarYandexstation2LinuxImageJIG: fastboot_img_path,
                qrt.QuasarYandexstation2BootloaderImageJIG: uboot_path},
            resources_attrs=dict(secureboot=secureboot, )
        )

    def exec_jig(self, secureboot, config):
        build_command = '. ./build/envsetup.sh && lunch {} && m {}'.format(
            self.BUILDROOT_CONFIG_JIG,
            ' '.join('{}={}'.format(k, v) for k, v in config.items()),
        )

        self.run_build(build_command)

        if secureboot:
            self.publish_jig(secureboot=True)
        self.publish_jig(secureboot=False)

    def exec_fp(self, secureboot, config):
        if secureboot:
            config['BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY'] = self.yav_export_file(
                str(self.Parameters.rabbit_hole_key_secret), 'rh_key.pem')
            config['BR2_PACKAGE_AVB_SUPPORT_AVB_KEY_PATH'] = self.yav_export_file(
                str(self.Parameters.avb_key_secret), 'avb_rsa2048.pem')
        else:
            config['BR2_TARGET_AMLOGIC_UBOOT_RABBIT_HOLE_KEY'] = ''
            config['BR2_PACKAGE_AVB_SUPPORT'] = 'n'

        build_command = '. ./build/envsetup.sh && lunch {} && m {}'.format(
            self.BUILDROOT_CONFIG_INS,
            ' '.join('{}={}'.format(k, v) for k, v in config.items()),
        )

        self.run_build(build_command)

        if secureboot:
            self.publish_fp(secureboot=True)
        self.publish_fp(secureboot=False)

    def exec_svc(self, secureboot, config):
        config['BR2_PACKAGE_PROVISION_PROTECTION_KEY'] = self.yav_export_file(
            str(self.Parameters.provision_protection_key_secret), 'pcpk.bin')

        build_command = '. ./build/envsetup.sh && lunch {} && m {}'.format(
            self.BUILDROOT_CONFIG_SVC,
            ' '.join('{}={}'.format(k, v) for k, v in config.items()),
        )

        self.run_build(build_command)

        if secureboot:
            self.publish_svc(secureboot=True)
        self.publish_svc(secureboot=False)

    def _on_execute(self):
        self._checkout()
        self._place_package_cache()

        secureboot = bool(self.Parameters.secureboot)

        if secureboot:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'y',
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN_KEYS_PATH': self.yav_export_tar(
                    str(self.Parameters.secureboot_keys_secret), 'tdk_keys.tgz'),

                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'y',
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN_KEY': self.yav_export_file(
                    str(self.Parameters.secureboot_keys_secret),
                    'aml-user-key.sig')
            }
        else:
            config = {
                'BR2_PACKAGE_TEE_SUPPORT_TA_SIGN': 'n',
                'BR2_TARGET_ROOTFS_AMLOGIC_SIGN': 'n'
            }

        if bool(self.Parameters.build_fp_img):
            self.exec_fp(secureboot, config)
        if bool(self.Parameters.build_jig_img):
            self.exec_jig(secureboot, config)
        if bool(self.Parameters.build_sl_img):
            self.exec_svc(secureboot, config)

        self._publish_repo_manifest()

    def yav_export_file(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of file to export
        :return str: path to exported file
        """
        return os.path.join(self.yav_export(secret_or_version, key), key)

    def yav_export_tar(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of tar archive to export
        :return str: path to exported dir
        """
        export_path = self.yav_export(secret_or_version, key)
        tar_path = os.path.join(export_path, key)
        extract_dir = tempfile.mkdtemp(prefix=key, dir=export_path)
        with tarfile.open(tar_path) as tar:
            tar.extractall(extract_dir)
        return extract_dir
