import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import client as ctc

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base
from sandbox.projects.quasar.image_builds.yandexstation2 import base as ys2_base
from sandbox.projects.quasar.platform import get_supported_targets_json


# These parts are removed from repo tag from left when forming version
# See cleanup_version
REPO_TAG_PREFIX_CLEANUPS = [ys2_base.PLATFORM_CODENAME, 'yandexstation_2', '-', '_']


def cleanup_version(version):
    """
    Cleans up version by removing parts of the last block
    that start from prefixes given in REPO_TAG_PREFIX_CLEANUPS
    """
    parts = version.split('.')

    tag = parts[-1]

    for prefix in REPO_TAG_PREFIX_CLEANUPS:
        if tag.startswith(prefix):
            tag = tag[len(prefix):]

    result = '.'.join(parts[:-1] + [tag])

    logging.info('Cleaned up version %s to %s' % (version, result))

    return result


class QuasarBuildYandexstation2Image(quasar_utils.YAVExportMixin,
                                     quasar_base.BaseQuasarImageBuildTask):
    """
    Build full image for quasar Station2 and publish it as a resource.
    """

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        quasar_platform = platform.Platform.YANDEXSTATION_2
        param_creator = quasar_base.ParameterCreator(quasar_platform)

        kill_timeout = 3 * 60 * 60  # 3h

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")
        custom_manifest = param_creator.create_custom_manifest_parameter()

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool('Suspend task for debug purposes just before actual build starts', default=False)

        ota_max_size = param_creator.create_ota_max_size_parameter()
        quasar_app = quasar_utils.LastStableResource('Quasar App .apk', resource_type=qrt.QuasarApp)
        quasar_daemons = param_creator.create_daemons_parameter()

        sign = sdk2.parameters.Bool('Build signed image and OTA', default=True)
        sign_keys_secret = sdk2.parameters.String('YAV secret with ota keys tarball, should have file keys.tgz.base64', default='sec-01e4vcfd4kwfe4nsc6t9cqfztw')

        factory = sdk2.parameters.Bool('Build factory image', default=False)

        hw_test_mode = sdk2.parameters.Bool('Build a hw test firmware, without quasar', default=False)

        secureboot = sdk2.parameters.Bool('Build secureboot image and OTA', default=True)
        secureboot_keys_secret = sdk2.parameters.String(
            'YAV secret with secureboot keys tarball, should have files: aml-user-key.sig, tdk_keys.tgz',
            default='ver-01e9qdk5tnqvzehjkw6p4xm1h9')
        rabbit_hole_key_secret = sdk2.parameters.String(
            'YAV secret with rabbit_hole rsa private key: rh_key.pem',
            default='ver-01ee2njeqfzasqzc2y1jtn3kkv')
        avb_key_secret = sdk2.parameters.String(
            'YAV secret with avb rsa private key: avb_rsa2048.pem',
            default='ver-01eectr8kn0ps2xdprvpz6aj9t')

        target_files = sdk2.parameters.Bool('Save target_files to resource', default=False)

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexstation2LxcImage)

    class Requirements(quasar_utils.YAVExportMixin.Requirements, quasar_base.BaseQuasarImageBuildTask.Requirements):
        client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.SSD  # repo is big
        disk_space = 200 * 1024  # 200 Gb

    class BuildConfig(ys2_base.YandexStation2BuildConfig):
        @property
        def daemons_dest_path(self):
            return self.task.path('daemons.in')

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def _on_execute(self):
        self._checkout()

        # abspath so _place / _publish wont try to prepend checkout dir
        ota_out, image_out, target_files_out, app_in = [
            os.path.abspath(str(self.path(p))) for p in
            ['ota.zip', 'aml.img', 'target_files.zip', 'app.apk.in']
        ]

        daemons_in = str(self.config.daemons_dest_path)
        self._place_daemons()

        # this unpacks app properly
        self._place_resources_to_location({
            self.Parameters.quasar_app: app_in,
        })

        # resource is unpacked read-only, but we need to modify it -- strip, write version, etc
        # there is sdk2.paths.add_write_permissions_for_path but it breaks other permissions =(
        sandbox_process.run_process(['chmod', '-R', 'u+w', daemons_in])
        sandbox_process.run_process(['chmod', '-R', 'u+w', app_in])

        version = cleanup_version(self._determine_version())

        self._place_version(version)

        build_args = [
            '--build_type', self.Parameters.build_type,
            '--build_version', version,
            '--daemons_path', daemons_in,
            '--quasar_app_path', app_in,
            '--image_dest', image_out,
            '--ota_dest', ota_out,
            '--verbose',
            '--clean',
        ]

        if self.Parameters.target_files:
            build_args += ['--target_files_dest', target_files_out]

        if self.Parameters.sign:
            build_args += ['--sign', '--sign_keys', self.yav_export_b64_file(self.Parameters.sign_keys_secret, 'keys.tgz.base64')]
            build_args += ['--tdk_key',
                           self.yav_export_file(str(self.Parameters.secureboot_keys_secret), 'tdk_keys.tgz')]

        if self.Parameters.build_type == 'user':
            build_args += ['--rabbit_hole_key',
                           self.yav_export_file(str(self.Parameters.rabbit_hole_key_secret), 'rh_key.pem')]

        if self.Parameters.secureboot:
            build_args += ['--secureboot', '--secureboot_key',
                           self.yav_export_file(str(self.Parameters.secureboot_keys_secret), 'aml-user-key.sig')]

        if bool(self.Parameters.avb_key_secret):
            avb_rsa_key = self.yav_export_file(
                str(self.Parameters.avb_key_secret), 'avb_rsa2048.pem')
            build_args += ['--avb-key', avb_rsa_key]

        if self.Parameters.factory:
            build_args += ['--factory']

        if self.Parameters.hw_test_mode:
            build_args += ['--hw_test_mode']

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Build args would be <%s>' % ' '.join(build_args))
            self.suspend()

        self._build(build_args)

        resources={
            qrt.QuasarYandexstation2Image: image_out,
            qrt.QuasarYandexstation2OTAImage: ota_out,
        }

        if self.Parameters.target_files:
            resources[qrt.QuasarYandexstation2TargetFiles] = target_files_out

        attrs = dict(
            buildtype=self.Parameters.build_type,
            version=version,
            signed=bool(self.Parameters.sign),
            factory=bool(self.Parameters.factory),
            secureboot=bool(self.Parameters.secureboot),
            hw_test_mode=bool(self.Parameters.hw_test_mode),
            repository_tag=self.repository_tag
        )

        targets_json = get_supported_targets_json(self.Parameters.quasar_platform,
                                                  self.Parameters.quasar_platform_revision)
        if targets_json:
            attrs['supported_targets'] = targets_json

        self._publish(
            resources=resources,
            resources_attrs=attrs
        )

        self._publish_repo_manifest()
