import os
import re
import logging
import tempfile
import tarfile

from datetime import date

from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk import process as sandbox_process

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import platform
from sandbox.projects.quasar import build_types
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base
from sandbox.projects.quasar.image_builds.yandexstation2 import base as ys2_base


class QuasarBuildYandexstation2ImageFromTargetFiles(quasar_utils.YAVExportMixin,
                                                    quasar_base.BaseQuasarImageBuildTask):
    """
    Build full image for quasar Station2 from target_files and publish it as a resource.
    """

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXSTATION_2)

        kill_timeout = 30 * 60  # 30m

        target_files = sdk2.parameters.Resource("Target files")

        quasar_app = quasar_utils.LastStableResource('Quasar App .apk', resource_type=qrt.QuasarApp)
        quasar_daemons = param_creator.create_daemons_parameter()

        sign = sdk2.parameters.Bool('Build signed image and OTA', default=True)
        sign_keys_secret = sdk2.parameters.String('YAV secret with ota keys tarball, should have file keys.tgz.base64', default='sec-01e4vcfd4kwfe4nsc6t9cqfztw')
        avb_key_secret = sdk2.parameters.String('YAV secret with avb rsa private key: avb_rsa2048.pem', default='ver-01eectr8kn0ps2xdprvpz6aj9t')

        otatools = quasar_utils.LastStableResource('OTA tools', resource_type=qrt.QuasarYandexstation2OtaTools, required=True)

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexstation2LxcImage)

    class BuildConfig(ys2_base.YandexStation2BuildConfig):
        @property
        def daemons_dest_path(self):
            return self.task.target_files_path / "VENDOR" / "quasar"

    def _check_input_resoruces(self):
        super(QuasarBuildYandexstation2ImageFromTargetFiles, self)._check_input_resoruces()

        target_files = self.Parameters.target_files
        if target_files is not None and target_files.buildtype != self.Parameters.build_type:
            raise errors.TaskFailure("target_files and task build type do not match")

    def _prepare_otatools(self):
        self.otatools_path = self.path("otatools")
        self.otatools_path.mkdir(parents=True)

        otatools_zip = sdk2.ResourceData(self.Parameters.otatools).path

        logging.info("Extracting otatools")
        sandbox_process.run_process(
            [
                "unzip",
                str(otatools_zip),
                "-d", str(self.otatools_path),
            ],
            log_prefix="extract_otatools",
        )

    def _prepare_target_files(self):
        self.target_files = self.Parameters.target_files

        if self.target_files is not None:
            logging.info("Using supplied target files resource")
            return

        attrs = {
            "buildtype": self.Parameters.build_type,
            "repository_tag": self.repository_tag,
        }

        logging.info("Looking for last target_files resource with attrs %s", str(attrs))

        self.target_files = sdk2.Resource.find(
            resource_type=qrt.QuasarYandexstation2TargetFiles,
            attrs=attrs,
        ).first()

        if self.target_files is None:
            raise errors.TaskFailure("target_files resource not found")

    def _determine_version(self):
        q_version = self._read_version_file(self.config.daemons_dest_path / "VERSION")

        tf_version = self.target_files.version.strip().split(".")

        tf_version[1] = q_version[0]
        tf_version[3] = q_version[1]
        tf_version[4] = str(self.id)
        tf_version[5] = date.today().strftime('%Y%m%d')

        version = '.'.join(tf_version)
        logging.info('determined version to be %s', version)

        return version

    def _prepare_vbmeta(self):
        misc_info_filename = "META/misc_info.txt"
        dt_img_filename = "IMAGES/dt.img"
        dtb_img_filename = "IMAGES/dtb-avb.img"
        logging.info("Unzip images and meta")
        sandbox_process.run_process(
            [
                "unzip",
                self.target_files_zip,
                dt_img_filename,
                misc_info_filename
            ],
            work_dir=str(self.target_files_path),
            log_prefix="zip_unzip",
        )
        os.rename(str(self.target_files_path / dt_img_filename), str(self.target_files_path / dtb_img_filename))

        sandbox_process.run_process(
            [
                str(self.otatools_path / "bin" / "avbtool"),
                "add_hash_footer",
                "--image", str(self.target_files_path / dtb_img_filename),
                "--partition_size", "262144",
                "--partition_name", "dtb"
            ],
            log_prefix="add_hash_footer",
        )

        avb_rsa_key = os.path.join(self.yav_export(self.Parameters.avb_key_secret, 'avb_rsa2048.pem'), 'avb_rsa2048.pem')

        with open(str(self.target_files_path / misc_info_filename), 'r+') as misc_info:
            info = re.sub(r'avb_vbmeta_key_path=.*', 'avb_vbmeta_key_path=' + avb_rsa_key, misc_info.read())
            misc_info.seek(0)
            misc_info.write(info)
            misc_info.truncate()

        return [
            dt_img_filename,
            misc_info_filename,
            "IMAGES/vbmeta.img"
        ]

    def _on_execute(self):
        self._prepare_target_files()
        self._prepare_otatools()

        self.target_files_path = self.path("target_files")
        self.target_files_path.mkdir(parents=True)

        self.ota_out, self.target_files_zip = [
            os.path.abspath(str(self.path(p))) for p in
            ['ota.zip', 'target_files.zip']
        ]
        app_in = str(self.target_files_path / "SYSTEM" / "priv-app" / "Quasar-App" / "Quasar-App.apk")

        self._place_daemons()

        self._place_resources_to_location({
            self.target_files: self.target_files_zip,
            self.Parameters.quasar_app: app_in,
        })

        self._add_write_permissions(self.target_files_zip)
        self._add_write_permissions(app_in)

        version = self._determine_version()
        self._place_version(version)

        self._build()

        # make _publish() happy
        self.repository_tag = self.config.default_repository_tag

        self._publish(
            resources={
                qrt.QuasarYandexstation2OTAImage: self.ota_out,
            },
            resources_attrs=dict(
                buildtype=self.Parameters.build_type,
                version=version,
                signed=bool(self.Parameters.sign),
                factory=False,
                secureboot=self.target_files.secureboot,
                hw_test_mode=False,
            )
        )

    def _build(self):
        delete_list = [
            "IMAGES/system.img",
            "IMAGES/system.map",
            "IMAGES/vendor.img",
            "IMAGES/vendor.map",
            "SYSTEM/priv-app/Quasar-App/Quasar-App.apk",
            "VENDOR/quasar/*",
        ]

        if self.Parameters.build_type != build_types.ImageBuildtype.ENGINEERING:
            delete_meta = self._prepare_vbmeta()
            delete_list.extend(delete_meta)

        logging.info("Removing old files from zip")
        sandbox_process.run_process(
            [
                "zip", "-d",
                self.target_files_zip,
            ] + delete_list,
            log_prefix="zip_delete",
        )

        logging.info("Creating quasar.cfg symlink")

        if self.Parameters.build_type == build_types.ImageBuildtype.ENGINEERING:
            quasar_cfg_name = "quasar-dev.cfg"
        else:
            quasar_cfg_name = "quasar-prod.cfg"

        sandbox_process.run_process(
            [
                "ln", "-s",
                quasar_cfg_name,
                str(self.target_files_path / "VENDOR" / "quasar" / "quasar.cfg"),
            ],
            log_prefix="link_quasar_cfg",
        )

        logging.info("Adding new files to zip")
        sandbox_process.run_process(
            [
                "zip",
                "-uryX",
                self.target_files_zip,
                ".",
            ],
            work_dir=str(self.target_files_path),
            log_prefix="zip_update",
        )

        logging.info("Rebuilding images")
        sandbox_process.run_process(
            [
                str(self.otatools_path / "releasetools" / "add_img_to_target_files"),
                "-a", "-v",
                "-p", str(self.otatools_path),
                self.target_files_zip,
            ],
            log_prefix="add_img_to_target_files",
        )

        logging.info("Building OTA")

        ota_args = [
            str(self.otatools_path / "releasetools" / "ota_from_target_files"),
            "-v",
            "--block",
            "-p", str(self.otatools_path),
            "--no_prereq",
        ]

        if self.Parameters.sign:
            sign_keys = self.yav_export_b64_file(self.Parameters.sign_keys_secret, 'keys.tgz.base64')

            keys_dir = tempfile.mkdtemp()
            with tarfile.open(sign_keys) as k:
                k.extractall(keys_dir)
                logging.info('Unpacked keys to %s', keys_dir)

            ota_args += ["-k", os.path.join(keys_dir, "releasekey")]

        ota_args += [self.target_files_zip, self.ota_out]

        sandbox_process.run_process(
            ota_args,
            log_prefix="ota_from_target_files",
        )
