import logging

from sandbox import sdk2
from sandbox.sandboxsdk import process as sandbox_process
from sandbox.common.types import client as ctc

from sandbox.projects.quasar import utils as quasar_utils
from sandbox.projects.quasar import resource_types as qrt
from sandbox.projects.quasar import platform
from sandbox.projects.quasar.image_builds.base_image_build import quasar as quasar_base
from sandbox.projects.quasar.image_builds.yandexstation2 import base as ys2_base


class QuasarBuildYandexstation2OtaTools(
        quasar_utils.YAVExportMixin,
        quasar_base.BaseQuasarImageBuildTask
):
    """
    Build OTA tools package for yandexstation_2 and publish it as a resource.
    """

    class Parameters(quasar_base.BaseQuasarImageBuildTask.Parameters):
        param_creator = quasar_base.ParameterCreator(platform.Platform.YANDEXSTATION_2)

        kill_timeout = 1 * 60 * 60  # 1h

        gerrit_inspections = sdk2.parameters.Dict(
            'Inspections in Gerrit to build with',
            description="Key-value pairs of inspection numbers and local paths"
                        " of projects relative to the repo's root")
        custom_manifest = param_creator.create_custom_manifest_parameter()

        with sdk2.parameters.Group('Internal') as internal_block:
            suspend_before_build = sdk2.parameters.Bool(
                'Suspend task for debug purposes just before actual build starts',
                default=False)

        _container = quasar_utils.LastStableContainer(
            'Execution container',
            resource_type=qrt.QuasarYandexstation2LxcImage)

        build_type = None

    class Requirements(quasar_utils.YAVExportMixin.Requirements,
                       quasar_base.BaseQuasarImageBuildTask.Requirements):
        client_tags = ctc.Tag.LINUX_BIONIC & ctc.Tag.SSD
        disk_space = 100 * 1024  # 100 Gb

    class BuildConfig(ys2_base.YandexStation2BuildConfig):
        def load_package_config(self):
            self.vcs = self.default_vcs
            self.repository_url = self.default_repository_url
            self.repository_tag = self.default_repository_tag

    @property
    def repo_overlay_inspections(self):
        return self.Parameters.gerrit_inspections

    def _on_execute(self):
        self._checkout()

        build_command = '. ./build/envsetup.sh && lunch yandexstation_2-eng && m otatools-package'

        if self.Parameters.suspend_before_build:
            logging.info('Suspending before build')
            logging.info('Execute command %s', build_command)
            self.suspend()

        sandbox_process.run_process(
            ['bash', '-c', build_command],
            log_prefix='build_otatools',
            work_dir=self.checkout_path,
        )

        otatools_path = "out/target/product/yandexstation_2/otatools.zip"

        self._publish(
            resources={
                qrt.QuasarYandexstation2OtaTools: otatools_path
            },
        )

        self._publish_repo_manifest()
