# This Python file uses the following encoding: utf-8
from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import binary_task
import json
import logging
from time import sleep


class KolhozDeviceConfigureTask(sdk2.Task, binary_task.LastBinaryTaskRelease):

    DEVICE_CONFIG_PATH = "/system/vendor/quasar/quasar.cfg"

    class Parameters(binary_task.LastBinaryReleaseParameters):
        device_id = sdk2.parameters.String("Device ID", required=True)
        kolhoz_id = sdk2.parameters.String("Kolhoz ID", required=True)
        tokens = sdk2.parameters.YavSecret("Tokens", default="sec-01dez1a5pfxech6v1h144k1kf4")
        model = sdk2.parameters.String("Model", default="yandexmini")

        target_version = sdk2.parameters.String("version", default="")
        version_id = sdk2.parameters.Integer("version_id")

        check_interval_sec = sdk2.parameters.Integer("Check interval sec", default=60)
        check_max_attermps = sdk2.parameters.Integer("Max check attempts", default=10)

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

    def build_request_url(self):
        return self.BACKEND_URL + self.Parameters.model + "/update"

    def update_version(self):
        from yandex_io.pylibs.quasmodrom_client import QuasmodromClient

        token = self.Parameters.tokens.data()["quasmodrom_token"]
        client = QuasmodromClient(token=token)
        try:
            client.set_device_settings(
                self.Parameters.model, self.Parameters.device_id, self.Parameters.version_id
            )
        except Exception as ex:
            raise common.errors.TaskFailure("Failed to create firmware update: {}".format(ex))

    def wait_device_update(self, device, version):
        for i in range(self.Parameters.check_max_attermps):
            try:
                device.reconnect()
                device.pull(self.DEVICE_CONFIG_PATH, str(self.path()))
                with open(str(self.path("quasar.cfg")), "r") as config:
                    device_version = json.load(config)["common"]["softwareVersion"]
                    logging.info(
                        "Version on device: {}, Target version: {}. Attempt {}".format(device_version, version, i + 1)
                    )
                    if device_version == version:
                        return
            except Exception as ex:
                logging.exception("Failed to check version: {}".format(ex))
            sleep(self.Parameters.check_interval_sec)
        raise common.errors.TaskFailure("Failed to update device to target version")

    def on_execute(self):
        from yandex_io.pylibs.kolhoz.kolhoz_api import KolhozDevice

        token = self.Parameters.tokens.data()["kolhoz_token"]

        device = KolhozDevice(self.Parameters.device_id, self.Parameters.kolhoz_id, token)
        device.connect()

        if self.Parameters.target_version:
            self.update_version()
            self.wait_device_update(device, self.Parameters.target_version)

    class Requirements(sdk2.Requirements):
        container_resource = 2694802101
