# This Python file uses the following encoding: utf-8
from sandbox.projects.quasar.kolhoz.run_tests import KolhozTestRun
from sandbox.projects.quasar.kolhoz.occupy_device import KolhozOccupyTask
from sandbox.projects.quasar.kolhoz.release_device import KolhozReleaseTask
from sandbox.projects.quasar.kolhoz.configure_device import KolhozDeviceConfigureTask
from sandbox.projects.common import binary_task
from sandbox.common.types import notification as ctn
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox import common

import logging


class KolhozSmartSpeakerRegression(sdk2.Task, binary_task.LastBinaryTaskRelease):
    class Parameters(sdk2.Parameters):
        binary_params = binary_task.binary_release_parameters(stable=True)
        model_name = sdk2.parameters.String("Model name", default="yandexmini")
        version = sdk2.parameters.String("Software version", default="")
        version_id = sdk2.parameters.Integer("Quasmodrom version id")
        testing_mode = sdk2.parameters.String("Tesing mode", default="regression")
        check_rollback = sdk2.parameters.Bool("Run rollback test", default=False)
        rollback_version = sdk2.parameters.String("Rollback Software version", default="")
        rollback_version_id = sdk2.parameters.Integer("Rollback Quasmodrom version id")
        tokens = sdk2.parameters.YavSecret("Tokens", default="sec-01dez1a5pfxech6v1h144k1kf4")
        ticket = sdk2.parameters.String("Ticket", default="")
        arcadia_url = sdk2.parameters.String("Arcadia url", default="arcadia:/arc/trunk/arcadia")
        notifications = [
            sdk2.Notification([ctt.Status.Group.BREAK, ctt.Status.Group.FINISH], ["slavashel"], ctn.Transport.EMAIL),
            sdk2.Notification([ctt.Status.Group.BREAK, ctt.Status.Group.FINISH], ["slavashel"], ctn.Transport.TELEGRAM),
        ]

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

    def _occupy_device(self):
        occupy_task = KolhozOccupyTask(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            model_name=self.Parameters.model_name,
            binary_executor_release_type=ctt.ReleaseStatus.STABLE,
        )
        occupy_task.save()
        occupy_task.enqueue()
        self.Context.occupy_task_id = occupy_task.id
        raise sdk2.WaitTask([occupy_task], ctt.Status.Group.FINISH, wait_all=True)

    def _configure_device(self, version, version_id):
        configure_task = KolhozDeviceConfigureTask(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            device_id=self.Context.device_id,
            kolhoz_id=self.Context.kolhoz_id,
            target_version=version,
            version_id=version_id,
            model=self.Parameters.model_name,
            binary_executor_release_type=ctt.ReleaseStatus.STABLE,
        )
        configure_task.save()
        configure_task.enqueue()
        return configure_task

    def _cleanup(self):
        release_device_task = KolhozReleaseTask(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            device_id=self.Context.device_id,
            kolhoz_id=self.Context.kolhoz_id,
            binary_executor_release_type=ctt.ReleaseStatus.STABLE,
        )
        release_device_task.save()
        release_device_task.enqueue()
        raise sdk2.WaitTask([release_device_task], ctt.Status.Group.FINISH, wait_all=True)

    def _test(self):
        test_task = KolhozTestRun(
            self,
            description="Child of {}".format(self.id),
            owner=self.owner,
            device_id=self.Context.device_id,
            kolhoz_device_id=self.Context.kolhoz_id,
            occupy_device=False,
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            definition_flags="-DREMOTE_TEST",
            test=True,
            test_threads=1,
            targets="yandex_io/functional_tests",
            test_params="remote_test=true use_tus=true testing_mode={}".format(self.Parameters.testing_mode),
            binary_executor_release_type=ctt.ReleaseStatus.STABLE,
            __requirements__={"container_resource": 2694802101},
        )
        test_task.save()
        test_task.enqueue()
        self.Context.test_task_id = test_task.id
        raise sdk2.WaitTask([test_task], ctt.Status.Group.FINISH, wait_all=True)

    def _rollback_test(self):
        rollback_version_id = self.Parameters.rollback_version_id
        rollback_version = self.Parameters.rollback_version
        if not rollback_version or not rollback_version_id:
            from yandex_io.pylibs.quasmodrom_client import QuasmodromClient

            token = self.Parameters.tokens.data()["quasmodrom_token"]
            client = QuasmodromClient(token)
            rollback_version_id, rollback_version = client.get_last_firmware_update(self.Parameters.model_name, "beta")
        task = self._configure_device(self.Parameters.rollback_version, self.Parameters.rollback_version_id)
        self.Context.rollback_task_id = task.id
        raise sdk2.WaitTask([task], ctt.Status.Group.FINISH, wait_all=True)

    def _send_to_startrek(self, ticket_id, test_task, rollback_task):
        import startrek_client

        token = self.Parameters.tokens.data()["startrek_token"]
        st_client = startrek_client.Startrek(token=token, useragent='robot-yandex-io')
        try:
            ticket = st_client.issues[ticket_id]
        except startrek_client.exceptions.Forbidden:
            logging.error("Failed to get task: {}".format(ticket_id))
            return
        text = """Kolhoz regression finished for firmware: {}, model: {}.
        Functional tests status: {}, task: https://sandbox.yandex-team.ru/task/{}/view
        Rollback test status: {}, task:  https://sandbox.yandex-team.ru/task/{}/view""".format(
            self.Parameters.version, self.Parameters.model_name, test_task.status, test_task.id, rollback_task.status, rollback_task.id
        )
        ticket.comments.create(text=text)

    def on_execute(self):
        with self.memoize_stage.occupy_stage():
            self._occupy_device()

        occupy_task = sdk2.Task[self.Context.occupy_task_id]
        if occupy_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Child occupy task failed")

        with self.memoize_stage.configure_stage():
            self.Context.device_id = occupy_task.Parameters.device_id
            self.Context.kolhoz_id = occupy_task.Parameters.kolhoz_id
            task = self._configure_device(self.Parameters.version, self.Parameters.version_id)
            self.Context.configure_task_id = task.id
            raise sdk2.WaitTask([task], ctt.Status.Group.FINISH, wait_all=True)

        configure_task = sdk2.Task[self.Context.configure_task_id]
        if configure_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Child configure task failed")

        with self.memoize_stage.test_stage():
            self._test()

        with self.memoize_stage.rollback_stge():
            if self.Parameters.check_rollback:
                self._rollback_test()
            else:
                self.Context.rollback_task_id = None

        with self.memoize_stage.cleanup_stage():
            self._cleanup()

        with self.memoize_stage.startrek_stage():
            ticket_id = self.Parameters.ticket
            if ticket_id:
                test_task = sdk2.Task[self.Context.test_task_id]
                rollback_task = sdk2.Task[self.Context.rollback_task_id]
                self._send_to_startrek(ticket_id, test_task, rollback_task)

        test_task = sdk2.Task[self.Context.test_task_id]
        if test_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Test task failed")
        if self.Context.rollback_task_id is None:
            return
        rollback_task = sdk2.Task[self.Context.rollback_task_id]
        if rollback_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure("Rollback task failed")
