import json

import sandbox.sdk2 as sdk2
import sandbox.common.types.client as ctc

import sandbox.projects.resource_types as resource_types
import sandbox.projects.common.binary_task as binary_task

from sandbox.projects.quasar.resource_types import QuasarDaemons, YandexIoPackageAnalysisReport


class YandexIoPackageAbiAnalyze(binary_task.LastBinaryTaskRelease, sdk2.Task):
    # Enable MULTISLOT execution: https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.MULTISLOT | ctc.Tag.GENERIC
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(binary_task.LastBinaryReleaseParameters):
        daemons_package = sdk2.parameters.Resource(
            "Package with YandexIO daemons",
            resource_type=QuasarDaemons,
            required=True,
        )

        with sdk2.parameters.Output:
            report = sdk2.parameters.Resource(
                "Package ABI analysis report",
                resource_type=YandexIoPackageAnalysisReport,
            )

    def on_execute(self):
        from smart_devices.ci.pylibs.package_abi.dir import extract_dir_abi
        from smart_devices.ci.pylibs.package_abi.elf import extract_elf_abi

        super(YandexIoPackageAbiAnalyze, self).on_execute()

        daemons_package_resource = self.Parameters.daemons_package
        daemons_package = sdk2.ResourceData(daemons_package_resource)

        # TODO: check package version?
        root_path = daemons_package.path / 'install_root'

        report_data = extract_dir_abi(root_path, extract_file_abi=extract_elf_abi)

        report_resource = YandexIoPackageAnalysisReport(
            self,
            "Package ABI report for sbr:{}".format(daemons_package_resource.id),
            'report.json',
        )
        report = sdk2.ResourceData(report_resource)

        self.Parameters.report = report_resource.id

        with open(str(report.path), 'w') as f:
            f.write(json.dumps(
                report_data,
                indent=2,
                sort_keys=True,
            ))
            f.write('\n')
        report.ready()


class YandexIoPackageAbiDiff(binary_task.LastBinaryTaskRelease, sdk2.Task):
    # Enable MULTISLOT execution: https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.MULTISLOT | ctc.Tag.GENERIC
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(binary_task.LastBinaryReleaseParameters):
        report_a = sdk2.parameters.Resource(
            "First (older) package ABI analysis report",
            resource_type=YandexIoPackageAnalysisReport,
            required=True,
        )
        report_b = sdk2.parameters.Resource(
            "Second (newer) package ABI analysis report",
            resource_type=YandexIoPackageAnalysisReport,
            required=True,
        )

    def on_execute(self):
        from smart_devices.ci.pylibs.package_abi.diff import diff_abi_trees

        super(YandexIoPackageAbiDiff, self).on_execute()

        self._report_lines = []

        report_a = sdk2.ResourceData(self.Parameters.report_a)
        with open(str(report_a.path), 'r') as f:
            report_a_data = json.loads(f.read())

        report_b = sdk2.ResourceData(self.Parameters.report_b)
        with open(str(report_b.path), 'r') as f:
            report_b_data = json.loads(f.read())

        has_diff = report_a_data != report_b_data
        if has_diff:
            diff_msg = '!!! FOUND DIFF!!!'
        else:
            diff_msg = 'OK, no diff.'

        self.Context.has_diff = has_diff
        self.Context.save()
        self._writeln(diff_msg)

        if has_diff:
            self._writeln('{}', diff_abi_trees(report_a_data, report_b_data))

        diff_report_resource = resource_types.PLAIN_TEXT(self, "", "diff_report.txt")
        diff_report = sdk2.ResourceData(diff_report_resource)
        with open(str(diff_report.path), 'w') as f:
            for line in self._report_lines:
                f.write(line)
                f.write('\n')
        diff_report.ready()

    def _writeln(self, fmt, *args, **kws):
        formatted_str = fmt.format(*args, **kws)
        self.set_info(formatted_str)
        self._report_lines.append(formatted_str)
