import sandbox.common.enum as sb_enum
import sandbox.common.format as sandbox_format
import sandbox.projects.quasar.build_types as build_types
import json


class Platform(sb_enum.Enum):
    """
    Device platforms

    See https://a.yandex-team.ru/arc/trunk/arcadia/smart_devices/platforms
    """
    sb_enum.Enum.preserve_order()
    sb_enum.Enum.lower_case()

    CENTAUR = None
    MAGRITTE = None
    ELARI_A98 = None
    JBL_LINK_MUSIC = None
    JBL_LINK_PORTABLE = None
    LIGHTCOMM = None  # here as it used by production metrics tasks
    LINKPLAY_A98 = None
    PRESTIGIO_SMART_MATE = None
    WK7Y = None
    X86_64 = None
    YANDEXMIDI = None
    YANDEXMINI = None
    YANDEXMINI_2 = None
    YANDEXMICRO = None
    YANDEXMODULE = None  # here as it used by production metrics tasks
    YANDEXSTATION = None
    YANDEXSTATION_2 = None
    YANDEXMODULE_2 = None
    PLUTO = None
    SATURN = None

    YANDEX_TV_HISI351_CVTE = None
    YANDEX_TV_MT6681_CV = None
    YANDEX_TV_MT6681_CVTE = None
    YANDEX_TV_MT9256_CVTE = None
    YANDEX_TV_MT9632_CV = None
    YANDEX_TV_MT9632_CVTE = None
    YANDEX_TV_MT9632_11_CVTE = None
    YANDEX_TV_RT2842_HIKEEN = None
    YANDEX_TV_RT2861_HIKEEN = None
    YANDEX_TV_RT2871_HIKEEN = None
    GOYA = None

_PLATFORMS_ADDITIONAL_REVISIONS = {
    Platform.YANDEXSTATION_2: ['rev2', 'rev3']
}


# YANDEXSTATION_2 rev1(None) will be built as mono-firmware for all revisions
_PLATFORMS_MONO_REVISIONS = {
    Platform.YANDEXSTATION_2: None
}


_PLATFORMS_PATH_KEY_FIXES = {
    Platform.JBL_LINK_MUSIC: 'jbl_link/music',
    Platform.JBL_LINK_PORTABLE: 'jbl_link/portable',
}


def get_platform_revisions(platform):
    revisions = _PLATFORMS_ADDITIONAL_REVISIONS.get(platform, [])
    revisions = [None] + revisions
    return revisions


def has_mono_revision(platform):
    return platform in _PLATFORMS_MONO_REVISIONS


def get_mono_revision(platform):
    return _PLATFORMS_MONO_REVISIONS[platform]


def get_effective_revisions(platform):
    if has_mono_revision(platform):
        return [get_mono_revision(platform)]
    else:
        return get_platform_revisions(platform)


# if platform has many revisions will return json string like:
#   [{"revision": "rev1"}, {"revision": "rev2"}, {"revision": "rev3"}]
def get_all_supported_targets_json(platform):
    revisions = get_platform_revisions(platform)
    if len(revisions) <= 1:
        return None

    targets = []
    for revision in revisions:
        target = {'revision': revision if revision is not None else 'rev1'}
        targets.append(target)
    return json.dumps(targets)


def get_supported_targets_json(platform, revision):
    if has_mono_revision(platform) and revision == get_mono_revision(platform):
        return get_all_supported_targets_json(platform)
    if revision:
        return '[{"revision": "' + revision + '"}]'
    return None


def get_platform_path_key(platform, revision=None):
    path_key = _PLATFORMS_PATH_KEY_FIXES.get(platform, platform)
    if revision is None:
        return path_key
    return path_key + '/' + revision


ANDROID_PLATFORMS = {
    Platform.YANDEXSTATION,
    Platform.YANDEXSTATION_2,
}

TV_PLATFORMS = {
    Platform.MAGRITTE,
    Platform.PLUTO,
    Platform.YANDEXMODULE_2,
    Platform.YANDEX_TV_HISI351_CVTE,
    Platform.YANDEX_TV_MT6681_CV,
    Platform.YANDEX_TV_MT6681_CVTE,
    Platform.YANDEX_TV_MT9256_CVTE,
    Platform.YANDEX_TV_MT9632_11_CVTE,
    Platform.YANDEX_TV_MT9632_CV,
    Platform.YANDEX_TV_MT9632_CVTE,
    Platform.YANDEX_TV_RT2842_HIKEEN,
    Platform.YANDEX_TV_RT2861_HIKEEN,
    Platform.YANDEX_TV_RT2871_HIKEEN,
    Platform.GOYA,
}

# Platforms that are no longer supported
RIP_PLATFORMS = {
    Platform.YANDEXMODULE,
    Platform.LIGHTCOMM,
    Platform.LINKPLAY_A98,
    Platform.ELARI_A98,
    Platform.PRESTIGIO_SMART_MATE,
}

PLATFORM_BUILD_TYPES = {
    Platform.JBL_LINK_MUSIC: list(build_types.LinkplayImageBuildType),
    Platform.JBL_LINK_PORTABLE: list(build_types.LinkplayImageBuildType),
    Platform.YANDEXSTATION: list(build_types.ImageBuildtype),
    Platform.YANDEXSTATION_2: list(build_types.ImageBuildtype),
    Platform.YANDEXMODULE_2: list(build_types.ImageBuildtype),
    Platform.PLUTO: list(build_types.ImageBuildtype),
}

APPS_BUILD_TYPES = {
    Platform.YANDEXMODULE_2: [
        build_types.TvAppsBuildType.RELEASE,
        build_types.TvAppsBuildType.QA,
        build_types.TvAppsBuildType.DEBUG,
    ],
    Platform.PLUTO: [
        build_types.TvAppsBuildType.RELEASE,
        build_types.TvAppsBuildType.QA,
        build_types.TvAppsBuildType.DEBUG,
    ],
    Platform.GOYA: [
        build_types.TvAppsBuildType.RELEASE,
        build_types.TvAppsBuildType.QA,
    ],
}

TV_APPS_FLAVOURED_PLATFORMS = {
    Platform.YANDEXMODULE_2,
    Platform.GOYA,
}

OTA_SIZE_LIMITS = {  # In megabytes, None means no check for ota size.
    Platform.ELARI_A98: 48,
    Platform.JBL_LINK_MUSIC: None,
    Platform.JBL_LINK_PORTABLE: None,
    Platform.LINKPLAY_A98: 48,
    Platform.PRESTIGIO_SMART_MATE: None,
    Platform.WK7Y: None,
    Platform.X86_64: None,
    Platform.YANDEXMIDI: None,
    Platform.YANDEXMINI: 80,
    Platform.YANDEXMINI_2: 70,
    Platform.YANDEXMICRO: 70,
    Platform.YANDEXMODULE_2: 1024,
    Platform.YANDEXSTATION: 1024,
    Platform.YANDEXSTATION_2: 1024,
    Platform.PLUTO: 1024,
    Platform.CENTAUR: 1024,
    Platform.MAGRITTE: 1024,
    Platform.SATURN: None,  # FIXME add real to size constraints
}

IMAGE_GENERATION_BLACKLIST = TV_PLATFORMS | RIP_PLATFORMS | {
    Platform.X86_64,
    Platform.WK7Y,
}

DAEMONS_BUILD_BLACKLIST = TV_PLATFORMS | RIP_PLATFORMS | {
    Platform.X86_64
}

NIGHTLY_WHITELIST = {
    Platform.YANDEXMINI,
    Platform.YANDEXMINI_2,
    Platform.YANDEXSTATION,
    Platform.YANDEXSTATION_2,
    Platform.JBL_LINK_MUSIC,
    Platform.JBL_LINK_PORTABLE,
    Platform.YANDEXMICRO,
    Platform.YANDEXMIDI,
}

UNIQ_DEVICE_KEYS_PLATFORMS = {
    Platform.YANDEXMICRO,
    Platform.YANDEXMIDI,
    Platform.YANDEXMINI_2,
    Platform.YANDEXSTATION,
    Platform.YANDEXSTATION_2,
    Platform.WK7Y,
}

PLATFORMS_WITH_FACTORY = {
    Platform.YANDEXMICRO,
    Platform.YANDEXMIDI,
    Platform.YANDEXMINI_2,
}

FIRMWARE_REPO_TRACKING = {
    Platform.YANDEXMICRO,
    Platform.YANDEXMIDI,
    Platform.YANDEXSTATION_2,
    Platform.YANDEXMINI_2,
    Platform.YANDEXMODULE_2,
    Platform.PLUTO,
    Platform.YANDEX_TV_MT6681_CV,
    Platform.YANDEX_TV_MT6681_CVTE,
    Platform.YANDEX_TV_MT9632_CV,
    Platform.YANDEX_TV_MT9632_CVTE,
    Platform.GOYA,
}

PLATFORMS_WITH_PERF = {
    Platform.YANDEXSTATION,
}

APPS_PLATFORMS = {
    Platform.CENTAUR
}

PLATFORMS_WITH_PERCENT_GROUP = {
    Platform.YANDEXMICRO,
    Platform.YANDEXMINI,
    Platform.YANDEXMINI_2,
}

PLATFORMS_IN_KOLHOZ = {
    Platform.YANDEXMINI,
    Platform.YANDEXMICRO
}


class TvPlatform():  # noqa
    PLATFORM_MAPPING = {
        Platform.GOYA: 'mt9632_cv',
    }

    # yandex_tv_mt9632_cv
    def __init__(self, quasar_platform):
        self.quasar_platform = quasar_platform
        platform_parts = (
            quasar_platform.split('_')[2:] if quasar_platform not in self.PLATFORM_MAPPING else
            self.PLATFORM_MAPPING[quasar_platform].split('_')
        )
        self.short = '_'.join(platform_parts) if quasar_platform not in self.PLATFORM_MAPPING else quasar_platform
        self.odm = platform_parts[-1]
        self.chipmaker = ''
        for character in platform_parts[0]:
            if character.isalpha():
                self.chipmaker += character
            else:
                break
        self.board_number = '_'.join([platform_parts[0][len(self.chipmaker):]] + platform_parts[1:-1])
        self.board = '_'.join(platform_parts[0:-1])
        self.sandbox_ident = sandbox_format.ident(self.short)
