from sandbox.common.format import ident
import sandbox.common.types.misc as ctm

from sandbox.projects.common.constants import constants as sandbox_constants

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.helpers.arcadia as rm_arc

import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows

from sandbox.projects.quasar.release_machine_jobs.common import (
    PlatformJobMixin,
    JobRunPlatformTasklet,
)
from sandbox.projects.quasar.platform import ANDROID_PLATFORMS, Platform


class QuasarBuildServicesJob(jg_build.JobGraphElementBuildBranched, PlatformJobMixin):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.BUILD

    @staticmethod
    def get_job_target():
        return 'services'

    def __init__(self, platform):
        out = {}
        if platform in ANDROID_PLATFORMS:
            out['QUASAR_APP'] = 'inf'

        super(QuasarBuildServicesJob, self).__init__(
            task_name='QUASAR_BUILD_SERVICES',
            build_item=self.get_job_name_parameter(platform),
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=20 << 30,  # in bytes
                    ),
                ),
            ),
            ctx=dict(
                vcs='SVN',
                platforms_to_build=platform,
                fill_output_param=True,
            ),
            out=out,
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            input_key='arcadia_url',
                            output_key='result_path',
                            transform=lambda x, params: rm_arc.construct_arc_url(x),
                        ),
                        jg_job_data.ParentDataOutput(
                            input_key='major_release_num',
                            output_key='branch_number_for_tag',
                        ),
                        jg_job_data.ParentDataOutput(
                            input_key='minor_release_num',
                            output_key='new_tag_number',
                        ),
                    ),
                ),
            ),
        )


class QuasarBuildServicesWithYaJob(jg_build.JobGraphElementBuildBranched, PlatformJobMixin):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.BUILD

    @staticmethod
    def get_job_target():
        return 'services'

    def get_packages(self, component_name, branch_number):
        if self.platform == Platform.JBL_LINK_MUSIC:
            package_platform = 'jbl_link/music'
        elif self.platform == Platform.JBL_LINK_PORTABLE:
            package_platform = 'jbl_link/portable'
        else:
            package_platform = self.platform

        base_path = 'smart_devices/platforms/{}/packages'.format(package_platform)
        return '{0}/daemons.json;{0}/symbols.json'.format(base_path)

    def __init__(self, platform):
        self.platform = platform

        super(QuasarBuildServicesWithYaJob, self).__init__(
            task_name='YA_PACKAGE',
            build_item=self.get_job_name_parameter(platform, ya_build=True),
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=30 << 30,  # in bytes
                        platform="linux_ubuntu_18.04_bionic",
                        hidden=False,
                    ),
                ),
            ),
            ctx=dict(
                build_type='minsizerel',
                package_type='tarball',
                raw_package=True,
                ignore_recurses=True,
                use_aapi_fuse=True,
                aapi_fallback=True,
                publish_package=False,
                resource_type='QUASAR_DAEMONS;QUASAR_SYMBOLS',
                clear_build=True,
                build_system='ya_force',  # force local build
                env_vars="QUASAR_PLATFORM_KEY_PATH='$(vault:file:QUASAR:quasar_daemons_{}_key)'".format(platform),
            ),
            out={
                'QUASAR_DAEMONS': 'inf',
            },
            job_arrows=(
                jg_arrows.ParentsData(
                    input_key='package_resource_attrs',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='major_release_num',
                                    output_key='branch_number_for_tag',
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key='minor_release_num',
                                    output_key='new_tag_number',
                                ),
                                jg_arrows.JustPassData(
                                    input_key="platform",
                                    value=self.platform,
                                ),
                            ),
                        ),
                    ),
                    transform=lambda major_release_num, minor_release_num, platform: dict(
                        quasar_platform=platform,
                        major_release_num=major_release_num,
                        minor_release_num=minor_release_num,
                    ),
                ),
                jg_arrows.ParentsData(
                    input_key='packages',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='component_name',
                                    output_key='component_name',
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key='branch_number',
                                    output_key='branch_number_for_tag',
                                ),
                            )
                        )
                    ),
                    transform=self.get_packages,
                )
            ),
        )


class QuasarBuildAppWithYaJob(jg_build.JobGraphElementBuildBranched, PlatformJobMixin):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.BUILD

    @staticmethod
    def get_job_target():
        return 'app'

    @staticmethod
    def get_packages(branch_number):
        return 'smart_devices/android/packages/apk/apk.json'

    def __init__(self, platform):
        super(QuasarBuildAppWithYaJob, self).__init__(
            task_name='YA_PACKAGE',
            build_item=self.get_job_name_parameter(platform, ya_build=True),
            job_params=dict(
                apiargs=dict(
                    requirements=dict(
                        disk_space=30 << 30,  # in bytes
                        dns=ctm.DnsType.DNS64,
                        platform="linux_ubuntu_18.04_bionic",
                        hidden=False,
                    ),
                ),
            ),
            ctx=dict(
                build_type='minsizerel',
                package_type='tarball',
                raw_package=True,
                ignore_recurses=True,
                use_aapi_fuse=True,
                aapi_fallback=True,
                publish_package=False,
                resource_type='QUASAR_APP',
                clear_build=True,
                build_system='ya_force',  # force local build
                env_vars='YANDEX_SIGNER_FILE=$(vault:file:QUASAR:robot-quasar-signer-oauth-token)',
            ),
            out={
                'QUASAR_APP': 'inf',
            },
            job_arrows=(
                jg_arrows.ParentsData(
                    input_key='package_resource_attrs',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='major_release_num',
                                    output_key='branch_number_for_tag',
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key='minor_release_num',
                                    output_key='new_tag_number',
                                ),
                            ),
                        ),
                        jg_arrows.JustPassData(
                            input_key="platform",
                            value=platform,
                        )
                    ),
                    transform=lambda platform, major_release_num, minor_release_num: dict(
                        quasar_platform=platform,
                        major_release_num=major_release_num,
                        minor_release_num=minor_release_num,
                    ),
                ),
                jg_arrows.ParentsData(
                    input_key='packages',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='branch_number',
                                    output_key='branch_number_for_tag',
                                ),
                            )
                        )
                    ),
                    transform=self.get_packages,
                )
            ),
        )


class QuasarBuildImageJob(jg_build.JobGraphElementBuildBranched, PlatformJobMixin):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.BUILD

    @staticmethod
    def get_job_target():
        return 'image'

    def __init__(self, platform, build_type=None, ya_build=False, sign=None, ya_build_apk=False):
        ctx = dict(
            keep_resources=True,
        )

        if build_type:
            if platform == Platform.YANDEXMINI:
                ctx['build_type_lplay'] = build_type
            else:
                ctx['build_type'] = build_type

        if sign is not None:
            ctx['sign'] = sign

        job_arrows = [
            QuasarBuildServicesWithYaJob.trigger(
                platform, None, ya_build, (
                    jg_job_data.ParentDataResource(
                        input_key='quasar_daemons',
                        resource_name='QUASAR_DAEMONS',
                    )
                ),
            ),
            jg_job_triggers.JobTriggerNewTag(
                parent_job_data=(
                    jg_job_data.ParentDataOutput(
                        input_key='quasar_minor_version',
                        output_key='new_tag_number',
                    ),
                    jg_job_data.ParentDataOutput(
                        input_key='quasar_major_version',
                        output_key='branch_number_for_tag',
                    ),
                ),
            ),
        ]

        if platform == Platform.YANDEXSTATION_2:
            job_arrows.append(jg_arrows.ValueAndParamsData(
                input_key='repository_config_dir',
                value='{svn_ssh_url}/arcadia/smart_devices/platforms/yandexstation_2/firmware@{revision}',
                transform=lambda y, x, rm_config: y.format(
                    svn_ssh_url=x.svn_ssh_url,
                    revision=x.revision,
                ),
            ))

        requirements = dict()
        if platform in ANDROID_PLATFORMS:
            apk_build_job = QuasarBuildAppWithYaJob if ya_build_apk else QuasarBuildServicesJob
            job_arrows.extend((
                apk_build_job.trigger(platform, None, ya_build_apk, (
                    jg_job_data.ParentDataResource(
                        input_key='quasar_app',
                        resource_name='QUASAR_APP',
                    )),
                ),
            ))

            if platform == Platform.YANDEXSTATION_2:
                ctx['factory'] = True

        super(QuasarBuildImageJob, self).__init__(
            task_name='QUASAR_BUILD_{}_IMAGE'.format(ident(platform)),
            build_item=self.get_job_name_parameter(platform, build_type, ya_build),
            ctx=ctx,
            job_arrows=job_arrows,
            job_params=dict(
                apiargs=dict(
                    requirements=requirements,
                ),
            ),
        )


class QuasarPublishToS3Job(JobRunPlatformTasklet):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.DEPLOY

    @staticmethod
    def get_tasklet_name():
        return 'PublishOtaToS3'

    @staticmethod
    def get_job_target():
        return 's3'

    def __init__(self, platform, build_type=None, ya_build=False, resource_filters=None):
        requirements = dict()
        if platform in ANDROID_PLATFORMS:
            requirements['disk_space'] = 5 << 30  # in bytes

        super(QuasarPublishToS3Job, self).__init__(
            job_type=self.get_job_type(),
            platform=platform,
            job_triggers=(
                QuasarBuildImageJob.trigger(platform, build_type, ya_build, (
                    jg_job_data.ParentDataOutput(
                        input_key='resources_ids',
                        output_key='ota_images',
                    )),
                ),
            ),
            frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
            build_type=build_type,
            ya_build=ya_build,
            job_params=dict(
                apiargs=dict(
                    requirements=requirements,
                ),
            ),
            extra_input=dict(
                resources_filters=resource_filters or {},
            ),
        )


class QuasarPublishToQuasmodromJob(JobRunPlatformTasklet):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.DEPLOY

    @staticmethod
    def get_tasklet_name():
        return 'OtaToQuasmodrom'

    @staticmethod
    def get_job_target():
        return 'quasmodrom'

    def __init__(self, platform, build_type=None, ya_build=False, add_ticket_in_comment=True):
        extra_input = dict(
            add_ticket_in_comment=add_ticket_in_comment,
            rm_platform=platform,
        )
        super(QuasarPublishToQuasmodromJob, self).__init__(
            job_type=self.get_job_type(),
            platform=platform,
            job_triggers=(
                QuasarPublishToS3Job.trigger(platform, build_type, ya_build, (
                    jg_job_data.ParentDataOutput(
                        input_key='resources_ids',
                        output_key='tasklet_output',
                        transform=lambda output, _: output['resourcesLinks'].keys(),
                    ),
                )),
                jg_job_triggers.JobTriggerNewTag((
                    jg_job_data.ParentDataOutput(
                        input_key='rm_component',
                        output_key='component_name',
                    ),
                    jg_job_data.ParentDataOutput(
                        input_key='rm_release_number',
                        output_key='branch_number_for_tag',
                    ),
                    jg_job_data.ParentDataOutput(
                        input_key='comment',
                        output_key='new_tag_number',
                        transform=lambda tag, params: '{}-{}: {}{}'.format(
                            params.db.db_name, tag, params.revision, ". Built with ya (and love)" if ya_build else ""
                        ),
                    ),
                )),
            ),
            extra_input=extra_input,
            frequency=jg_utils.TestFrequency.CHECK_EACH_COMMIT,
            build_type=build_type,
            ya_build=ya_build,
        )


class QuasarYaMakeMinusTJob(jg_test.JobGraphElementTestBranchCommon):
    @staticmethod
    def get_targets(component_name, branch_number):
        return 'yandex_io/services'

    def __init__(self):
        super(QuasarYaMakeMinusTJob, self).__init__(
            task_name='YA_MAKE',
            platform="linux_ubuntu_18.04_bionic",
            job_params={
                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
            },
            ctx={
                sandbox_constants.USE_AAPI_FUSE: True,
                sandbox_constants.ALLOW_AAPI_FALLBACK: True,
                sandbox_constants.CHECKOUT_MODE: sandbox_constants.CHECKOUT_MODE_AUTO,
                sandbox_constants.CHECKOUT: True,
                sandbox_constants.BUILD_SYSTEM_KEY: 'ya',
                sandbox_constants.CLEAR_BUILD_KEY: False,
                sandbox_constants.CHECK_RETURN_CODE: True,
                sandbox_constants.TESTS_REQUESTED: True,
                sandbox_constants.TEST_THREADS: 1,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            input_key=sandbox_constants.ARCADIA_URL_KEY,
                            output_key='result_path',
                            transform=lambda x, params: rm_arc.construct_arc_url(x),
                        ),
                    ),
                ),
                jg_arrows.ParentsData(
                    input_key='targets',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='component_name',
                                    output_key='component_name',
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key='branch_number',
                                    output_key='branch_number_for_tag',
                                ),
                            )
                        )
                    ),
                    transform=self.get_targets,
                )
            ),
        )
