from sandbox.common.format import ident
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release

from sandbox.projects.quasar.release_machine_jobs.build_stage import (
    QuasarBuildServicesJob,
    QuasarBuildServicesWithYaJob,
    QuasarBuildAppWithYaJob,
    QuasarBuildImageJob,
    QuasarPublishToS3Job,
    QuasarPublishToQuasmodromJob,
)
from sandbox.projects.quasar.release_machine_jobs.release_stage import (
    QuasarCreateStartrekTicketJob,
    QuasarUpdateStartrekTicketJob,
    QuasarCreateQuasmodromUpdateJob,
    QuasarActionReleaseToQuasmodrom,
)
from sandbox.projects.quasar.platform import ANDROID_PLATFORMS


def ota_filters(sign):
    if sign:
        return dict(signed='True')
    else:
        return None


class BuildConfig(object):
    """
    Constrains:
    - Only one build with release = True
    - Only one build with push_to_ticket = True (for each build_type for android component)
    """
    @property
    def ota_filters(self):
        return ota_filters(self.sign_image)

    def __init__(
            self,
            use_ya,
            build_type=None,
            sign_image=None,
            push_to_ticket=True,
            release=False,
            ya_build_apk=True,
    ):
        self.use_ya = use_ya
        self.build_type = build_type
        self.sign_image = sign_image
        self.push_to_ticket = push_to_ticket
        self.release = release
        self.ya_build_apk = ya_build_apk


class PlatformConfig(object):
    def __init__(self, component, platform, build_configs):
        self.component = component
        self.platform = platform
        self.builds = build_configs

    @property
    def ota_resource(self):
        return 'QUASAR_{}_OTAIMAGE'.format(ident(self.platform))

    @property
    def daemons_resource(self):
        return 'QUASAR_{}_DAEMONS'.format(ident(self.platform))

    @property
    def app_resource(self):
        return 'QUASAR_{}_APP'.format(ident(self.platform))

    @property
    def ya_build(self):
        for config in self.builds:
            if config.use_ya:
                return True
        return False

    @property
    def non_ya_build_apk(self):
        if self.platform not in ANDROID_PLATFORMS:
            return False

        for config in self.builds:
            if not config.ya_build_apk:
                return True
        return False

    @property
    def to_publish(self):
        for config in self.builds:
            if config.push_to_ticket:
                yield config

    @property
    def to_release(self):
        for config in self.builds:
            if config.release:
                return config
        return None

    @property
    def branch_part(self):
        branch_part = []

        if self.ya_build:
            branch_part.append(QuasarBuildServicesWithYaJob(self.platform))

            if self.platform in ANDROID_PLATFORMS:
                branch_part.append(QuasarBuildAppWithYaJob(self.platform))

        if self.non_ya_build_apk:
            branch_part.append(QuasarBuildServicesJob(self.platform))

        for build in self.builds:
            branch_part.extend([
                QuasarBuildImageJob(self.platform, build.build_type, build.use_ya, build.sign_image, build.ya_build_apk),
                QuasarPublishToS3Job(self.platform, build.build_type, build.use_ya, build.ota_filters),
                QuasarPublishToQuasmodromJob(self.platform, build.build_type, build.use_ya),
            ])

        return branch_part

    @property
    def release_part(self):
        release_part = [
            QuasarCreateStartrekTicketJob(self.platform, self.ota_resource),
        ]

        release_build = self.to_release
        if release_build:
            stages = [rm_const.ReleaseStatus.testing]
            if self.component == rm_const.RMNames.QUASAR_LINUX:  # TODO: remove
                stages += [
                    rm_const.ReleaseStatus.prestable,
                    rm_const.ReleaseStatus.stable,
                ]

            for release_stage in stages:
                def resource_ids_transform(
                    daemons_res_id,
                    ota_res_id,
                    app_res_id,
                    daemons_resource,
                    ota_resource,
                    app_resource,
                ):
                    result = {
                        daemons_resource: daemons_res_id,
                        ota_resource: ota_res_id,
                    }
                    if app_res_id is not None:
                        result[app_resource] = app_res_id
                    return result

                release_part.append(jg_release.JobGraphElementReleaseBranched(
                    release_to=release_stage,
                    release_item=self.platform,
                    job_arrows=[
                        jg_arrows.ParentsData(
                            input_key='component_resources',
                            triggers=(
                                QuasarBuildServicesWithYaJob.trigger(
                                    self.platform,
                                    None,
                                    release_build.use_ya,
                                    (
                                        jg_job_data.ParentDataResource(
                                            input_key='daemons_res_id',
                                            resource_name='QUASAR_DAEMONS',
                                        )
                                    ),
                                ),
                                QuasarPublishToQuasmodromJob.trigger(
                                    self.platform,
                                    release_build.build_type,
                                    release_build.use_ya,
                                    (
                                        jg_job_data.ParentDataOutput(
                                            input_key='ota_res_id',
                                            output_key='tasklet_output',
                                            transform=lambda output, _:
                                            output['resourcesLinks'].items()[0][0],
                                        )
                                    ),
                                ),
                                QuasarBuildAppWithYaJob.trigger(
                                    self.platform,
                                    None,
                                    release_build.use_ya,
                                    jg_job_data.ParentDataResource(
                                        input_key='app_res_id',
                                        resource_name='QUASAR_APP',
                                    ),
                                ) if self.platform in ANDROID_PLATFORMS else jg_arrows.JustPassData(
                                    input_key="app_res_id",
                                    value=None,
                                ),
                                jg_arrows.JustPassData(
                                    input_key="ota_resource",
                                    value=self.ota_resource,
                                ),
                                jg_arrows.JustPassData(
                                    input_key="daemons_resource",
                                    value=self.daemons_resource,
                                ),
                                jg_arrows.JustPassData(
                                    input_key="app_resource",
                                    value=self.app_resource,
                                ),
                            ),
                            transform=resource_ids_transform,
                        ),
                    ] + [
                        QuasarUpdateStartrekTicketJob.trigger(self.platform, build.build_type, build.use_ya)
                        for build in self.to_publish
                    ],
                ))

            release_part.extend(
                QuasarUpdateStartrekTicketJob(self.platform, self.component, build.build_type, build.use_ya)
                for build in self.to_publish
            )

            release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                release_to=rm_const.ReleaseStatus.testing,
                release_item=self.platform,
            ))

            if self.component == rm_const.RMNames.QUASAR_LINUX:  # TODO: remove
                for stage, group, critical in [
                    (rm_const.ReleaseStatus.prestable, 'beta', False),
                    (rm_const.ReleaseStatus.stable, 'default', False),
                ]:
                    release_part.extend([
                        QuasarCreateQuasmodromUpdateJob(self.platform, group, critical, release_build.build_type),
                        QuasarActionReleaseToQuasmodrom(stage, self.platform, group, critical),
                    ])

        return release_part
