import sandbox.projects.release_machine.core.const as rm_const

import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jg_element
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release

from sandbox.projects.quasar.release_machine_jobs.common import (
    PlatformJobMixin,
    JobRunPlatformTasklet,
)
from sandbox.projects.quasar.release_machine_jobs.build_stage import (
    QuasarPublishToQuasmodromJob,
)


class QuasarCreateStartrekTicketJob(jg_element.JobGraphElement, PlatformJobMixin):
    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.STARTREK

    @staticmethod
    def get_job_target():
        return 'create'

    def __init__(self, platform, release_item):
        super(QuasarCreateStartrekTicketJob, self).__init__(
            job_params=dict(
                job_type=self.get_job_type(),
                job_name_parameter=self.get_job_name_parameter(platform),
                task_name='CREATE_PLATFORM_STARTREK_TICKET',
                should_add_to_db=jg_utils.should_add_to_db_branch,
                apiargs=dict(
                    hidden=False,
                    requirements=dict(
                        disk_space=2 * (1024 ** 3),  # in Bytes
                    ),
                ),
                ctx=dict(
                    platform=platform,
                    release_item=release_item,
                    update_issue_if_exists=False,
                )
            ),
            job_arrows=(
                jg_arrows.ParamsData(
                    input_key='additional_info',
                    transform=lambda params, _: params.custom_fields.get('additional_info', '') + '\n'.join([
                        'Intermediate db name: {}'.format(params.db.db_name),
                    ]),
                ),
                jg_job_triggers.JobTriggerChangelogFinal(
                    parent_job_data=(
                        jg_job_data.ParentDataResource('changelog', 'RELEASE_MACHINE_CHANGELOG'),
                    ),
                ),
            ),
        )


class QuasarUpdateStartrekTicketJob(JobRunPlatformTasklet):
    @classmethod
    def trigger(cls, platform, build_type=None, ya_build=False, parent_job_data=()):
        # ignore ya_build since tasklet only supports multiple build_type's
        return super(QuasarUpdateStartrekTicketJob, cls).trigger(platform, build_type, False, parent_job_data)

    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.STARTREK

    @staticmethod
    def get_tasklet_name():
        return 'FirmwareToSt'

    @staticmethod
    def get_job_target():
        return 'update'

    def __init__(self, platform, component_name, build_type=None, ya_build=False):
        super(QuasarUpdateStartrekTicketJob, self).__init__(
            job_type=self.get_job_type(),
            platform=platform,
            job_triggers=(
                jg_job_triggers.JobTriggerNewTag(
                    parent_job_data=(
                        jg_job_data.ParentDataOutput(
                            input_key='tag_number',
                            output_key='new_tag_number',
                        ),
                        jg_job_data.ParentDataOutput(
                            input_key='branch_number',
                            output_key='branch_number_for_tag',
                        ),
                    ),
                ),
                QuasarCreateStartrekTicketJob.trigger(platform),
                QuasarPublishToQuasmodromJob.trigger(platform, build_type, ya_build, (
                    jg_job_data.ParentDataOutput(
                        input_key='firmware_link',
                        output_key='tasklet_output',
                        transform=lambda output, _: output['resourcesLinks'].items()[0][1],
                    )),
                )
            ),
            extra_input=dict(
                component_name=component_name,
                build_type=build_type,
                platform=platform,
            ),
            build_type=build_type,
        )


class QuasarCreateQuasmodromUpdateJob(JobRunPlatformTasklet):
    @classmethod
    def trigger(cls, platform, group_name=None, critical=False , build_type=None, ya_build=False, parent_job_data=()):
        return jg_arrows.JobTrigger(
            job_type=cls.get_job_type(),
            job_name_parameter=cls.get_job_name_parameter(platform, build_type, ya_build, group_name, critical),
            parent_job_data=parent_job_data,
        )

    @staticmethod
    def get_job_type():
        return rm_const.JobTypes.RELEASE

    @staticmethod
    def get_tasklet_name():
        return 'UpdateToQuasmodrom'

    @staticmethod
    def get_job_target():
        return 'QUASMODROM'

    @classmethod
    def get_job_name_parameter(cls, platform, build_type=None, ya_build=False, group_name=None, critical=False):
        modifiers = [platform, cls.get_job_target()]

        if group_name:
            modifiers.append(group_name)

        if critical:
            modifiers.append('critical')

        return '_'.join(modifiers)

    def __init__(self, platform, group_name, critical, build_type=None, ya_build=True):
        super(QuasarCreateQuasmodromUpdateJob, self).__init__(
            job_type=self.get_job_type(),
            job_params=dict(
                job_name_parameter=self.get_job_name_parameter(platform, build_type, ya_build, group_name, critical),
            ),
            platform=platform,
            job_triggers=(
                jg_job_triggers.JobTriggerNewTag(),
                jg_arrows.ParentsData(
                    input_key='comment',
                    triggers=(
                        jg_job_triggers.JobTriggerNewTag(
                            parent_job_data=(
                                jg_job_data.ParentDataOutput(
                                    input_key='major_release_num',
                                    output_key='branch_number_for_tag',
                                ),
                                jg_job_data.ParentDataOutput(
                                    input_key='minor_release_num',
                                    output_key='new_tag_number',
                                ),
                            ),
                        ),
                    ),
                    transform=lambda major_release_num, minor_release_num: 'Release {}-{}'.format(
                        major_release_num, minor_release_num
                    ),
                ),
                QuasarPublishToQuasmodromJob.trigger(platform, build_type, ya_build, (
                    jg_job_data.ParentDataOutput(
                        input_key='firmware_id',
                        output_key='tasklet_output',
                        transform=lambda output, _: output['firmwares'].values()[0]['id'],
                    )),
                ),
                QuasarCreateStartrekTicketJob.trigger(platform, parent_job_data=(
                    jg_job_data.ParentDataOutput(
                        input_key='ticket',
                        output_key='startrek_issue',
                    ),
                )),
            ),
            extra_input=dict(
                group_name=group_name,
                critical=critical,
            ),
            build_type=build_type,
        )


class QuasarActionReleaseToQuasmodrom(jg_release.JobGraphElementActionReleaseBase):
    def __init__(self, release_to, platform, group_name, critical):
        release_name_parameter = "{}__{}".format(platform, release_to) if platform else release_to
        job_name_parameter = "{}__critical".format(release_name_parameter) if critical else release_name_parameter

        super(QuasarActionReleaseToQuasmodrom, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params={
                "should_add_to_db": jg_utils.should_add_to_db_branch,
                "cancel_fallbehind_runs_on_fix": False,
            },
            job_arrows=(
                jg_job_triggers.JobTriggerRelease(job_name_parameter=release_name_parameter),
                QuasarCreateQuasmodromUpdateJob.trigger(platform, group_name, critical),
            ),
        )
