import sandbox.common as sandbox_common
import sandbox.projects.common.binary_task as binary_task
import sandbox.sdk2 as sdk2

import logging
import multiprocessing
import os
import shutil
import subprocess


def safe_run_repo_tool(*args, **kwargs):
    try:
        subprocess.check_call(*args, **kwargs)
    except subprocess.CalledProcessError as e:
        raise sandbox_common.errors.TaskError('repo exited with code {}, {}'.format(e.returncod, e.message))


class RepoManifestDetalization(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(binary_task.LastBinaryReleaseParameters):
        repo_manifest_url = sdk2.parameters.String('Url for repo init')
        branch = sdk2.parameters.String('Branch for repo init')
        depth = sdk2.parameters.Integer('Clone depth for repo init', default=1)
        ssh_user = sdk2.parameters.String('Ssh user name')
        ssh_key_vault_owner = sdk2.parameters.String('Sandbox vault item owner for ssh key')
        ssh_key_vault_name = sdk2.parameters.String('Sandbox vault item name for ssh key')
        do_commit = sdk2.parameters.Bool('Commit detailed manifest state to arcadia', default=False)
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Manifest path in arcadia (arcadia:/arc/trunk/arcadia/path_in_arcadia)')

    def on_execute(self):
        super(RepoManifestDetalization, self).on_execute()
        manifest_filename = 'branch_{}_manifest.xml'.format(self.Parameters.branch)

        with sdk2.ssh.Key(
            self,
            key_owner=self.Parameters.ssh_key_vault_owner,
            key_name=self.Parameters.ssh_key_vault_name
        ):
            with sdk2.helpers.ProcessLog(self, logger='repo_init') as pl:
                repo_init_list = [
                    'repo', 'init',
                    '-u', 'ssh://{}@{}'.format(self.Parameters.ssh_user, self.Parameters.repo_manifest_url),
                    '-b', self.Parameters.branch,
                ]
                if self.Parameters.depth:
                    repo_init_list.append('--depth')
                    repo_init_list.append(str(self.Parameters.depth))

                logging.info('Running %s', ' '.join(repo_init_list))
                safe_run_repo_tool(repo_init_list, stdout=pl.stdout, stderr=pl.stderr)
            with sdk2.helpers.ProcessLog(self, logger='repo_sync') as pl:
                repo_sync_list = ['repo', 'sync', '-j', str(multiprocessing.cpu_count())]
                logging.info('Running %s', ' '.join(repo_sync_list))
                safe_run_repo_tool(repo_sync_list, stdout=pl.stdout, stderr=pl.stderr)
            with sdk2.helpers.ProcessLog(self, logger='repo_manifest') as pl:
                repo_manifest_list = ['repo', 'manifest', '-r', '-o', manifest_filename]
                logging.info('Running %s', ' '.join(repo_manifest_list))
                safe_run_repo_tool(repo_manifest_list, stdout=pl.stdout, stderr=pl.stderr)

            with open(manifest_filename) as manifest_file:
                manifest_data = manifest_file.read()
                logging.debug(manifest_data)

            arcadia_manifest_dirname = os.path.basename(str(self.Parameters.arcadia_url).rstrip('/'))
            if self.Parameters.do_commit:
                sdk2.svn.Arcadia.checkout(
                    url=self.Parameters.arcadia_url,
                    path=arcadia_manifest_dirname,
                    depth='immediates'
                )
                arcadia_manifest_filename = os.path.join(arcadia_manifest_dirname, manifest_filename)
                with open(arcadia_manifest_filename) as arcadia_manifest_file:
                    arcadia_manifest_data = arcadia_manifest_file.read()

                if manifest_data != arcadia_manifest_data:
                    shutil.copyfile(manifest_filename, arcadia_manifest_filename)
                    sdk2.svn.Arcadia.commit(
                        path=arcadia_manifest_dirname,
                        message='Sync {} changes SKIP_CHECK'.format(self.Parameters.repo_manifest_url),
                        user=self.Parameters.ssh_user,
                    )
