"""
Holds resource types used in quasar -- images, packages, etc
"""
import sandbox.projects.quasar.users as quasar_users
from sandbox import sdk2
from sandbox.projects.quasar.platform import Platform
from sandbox.sdk2.service_resources import LxcContainer


class QuasarImageAndroidCcacheTarball(sdk2.Resource):
    """
    Android's cchache directory in a tarball.
    Pull it before building android part of the image.

    See https://source.android.com/setup/initializing#setting-up-ccache
    """
    ttl = 90
    auto_backup = True


class QuasarGradleCacheTarball(sdk2.Resource):
    """
    Cache of gradle resources to be carried around, in a tarball
    Precisely, contents of `GRADLE_USER_HOME` directory
    """
    ttl = 90
    auto_backup = True


class QuasarResourceBase(sdk2.Resource):
    ttl = 180
    auto_backup = True
    releasable = True


class QuasarLxcImageBase(LxcContainer):
    ttl = "inf"
    auto_backup = True
    releasable = True
    releasers = quasar_users.system_images_releasers

    __default_attribute__ = sdk2.parameters.String


class QuasarYandexSmartDeviceLxcImage(QuasarLxcImageBase):
    """
    Common LXC image to run android/sideload linux image building tasks for all smart_devices.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class QuasarYandexmidiLxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station Mini.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class QuasarLinkplayA98LxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station Mini.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class QuasarYandexminiLxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station Mini.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class QuasarYandexmini2LxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station Mini 2.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#use-custom-lxc-image
    """


class QuasarYandexmicroLxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex Micro

    See https://wiki.yandex-team.ru/sandbox/cookbook/#use-custom-lxc-image
    """


class QuasarYandexstationLxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class QuasarYandexstation2LxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for Yandex.Station2.

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """


class _QuasarImageBase(QuasarResourceBase):
    buildtype = sdk2.parameters.String('BUILDTYPE for this image')
    version = sdk2.parameters.String('Full version of the software')
    signed = sdk2.parameters.Bool('If this image is production-signed')


class _QuasarOTABase(_QuasarImageBase):
    """
    A base class for all OTA images. It declares attributes required for s3 upload / quasrmodrom / updatesd

    See `arcadia/quasar/infra/tasklets/publish_ota_to_s3`
    """
    platform = None  # NB: children should declare platform property with sdk2 parameter with default value
    supported_targets = sdk2.parameters.String('Supported targets tags')
    s3_url = sdk2.parameters.String('URL to our S3 where this is published')
    quasmodrom_id = sdk2.parameters.String('Quasmodrom firmware id of ota image')
    crc32_checksum = sdk2.parameters.String('CRC32 checksum of this resource')

    major_release_num = sdk2.parameters.Integer('Major release number', default=None)
    minor_release_num = sdk2.parameters.Integer('Minor release number', default=None)
    do_not_publish_me = sdk2.parameters.Bool('Do not publish ota to s3 and quasmodrom.')
    arcanum_review_id = sdk2.parameters.String('Arcanum review ID')


class QuasarYandexstationImage(_QuasarImageBase):
    """
    A full image for Yandex.Station redy to be flashed
    """


class QuasarYandexstationOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Yandex.Station
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.YANDEXSTATION)


class QuasarYandexstationTargetFiles(_QuasarImageBase):
    """
    A `target_files.zip` file from AOSP build process -- used in later stages, for Yandex.Station
    """


# Yandex.Chiron
# Full platform image.
class QuasarChironImage(_QuasarImageBase):
    """
    A full image for Yandex.Chiron.
    """
    factory = sdk2.parameters.Bool('If this image contains factory reset partition')
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    hw_test_mode = sdk2.parameters.Bool(
        'If this image is for hardware-testing ONLY (NOT SUITABLE FOR PRODUCTION DEVICES), see QUASAR-7125')
    verbose_version = sdk2.parameters.String('A `str` version for this build')


# OTA image.
class QuasarChironOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Yandex.Chiron
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.CENTAUR)
    factory = sdk2.parameters.Bool('If this image contains factory reset partition')
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    verbose_version = sdk2.parameters.String('A `str` version for this build')


class QuasarChironTargetFiles(_QuasarImageBase):
    """
    A `target_files.zip` file from AOSP build process -- used in later stages, for Yandex.Chiron
    """
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    repository_tag = sdk2.parameters.String('Repository tag')


# Centaur artifacts config
class QuasarCentaurArtifactsConfig(QuasarResourceBase):
    """
    Centaur Apps artifacts YAML config
    """


# TV artifacts config
class QuasarCentaurArtifacts(QuasarResourceBase):
    """
    Centaur Apps artifacts
    """


class QuasarLinkplayA98OTAImage(_QuasarOTABase):
    """
    An OTA-ready package for LinkplayA98
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.LINKPLAY_A98)


class QuasarElariA98OTAImage(_QuasarOTABase):
    """
    An OTA-ready package for ElariA98
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.ELARI_A98)


class QuasarPrestigioSmartMateOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Prestigio Smart Mate
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.PRESTIGIO_SMART_MATE)


# Yandex.Station Mini
# Full platform image.
class QuasarYandexminiImage(_QuasarImageBase):
    """
    A full image for Yandex.Station Mini.
    """


# OTA image.
class QuasarYandexminiOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for YandexMini (packed)
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.YANDEXMINI)


# repo menifest
class QuasarRepoManifest(QuasarResourceBase):
    """
    Repo manifest file
    """


# Yandex.Station Mini 2
# Full platform image.
class QuasarYandexmini2Image(_QuasarImageBase):
    """
    A full image for Yandex.Station Mini 2.
    """


# Yandex.Station Mini 2
# OTA image.
class QuasarYandexmini2OTAImage(_QuasarOTABase):
    """
    An OTA-ready package for YandexMini2 (packed)
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.YANDEXMINI_2)


# Yandex.Station Mini 2
# EFUSE file.
class QuasarYandexmini2Efuse(QuasarResourceBase):
    """
    EFUSE file for YandexMini2 securebooting
    """


# Yandex Midi
# Full platform image.
class QuasarYandexmidiImage(_QuasarImageBase):
    """
    A full image for Yandex Midi.
    """


# Yandex Midi
# OTA image.
class QuasarYandexmidiOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Yandex Midi (packed)
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.YANDEXMIDI)


# Saturn
# Full platform image.
class QuasarSaturnImage(_QuasarImageBase):
    """
    A full image for Saturn.
    """


# Saturn
# OTA image.
class QuasarSaturnOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Saturn (packed)
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.SATURN)


# Yandex Micro
# Full platform image.
class QuasarYandexmicroImage(_QuasarImageBase):
    """
    A full image for Yandex Micro.
    """


# Yandex Micro
# OTA image.
class QuasarYandexmicroOTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Yandex Micro (packed)
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.YANDEXMICRO)


# Yandex Micro
# EFUSE file.
class QuasarYandexmicroEfuse(QuasarResourceBase):
    """
    EFUSE file for Yandex Micro securebooting
    """


# Yandex.Station2
# Full platform image.
class QuasarYandexstation2Image(_QuasarImageBase):
    """
    A full image for Yandex.Station2.
    """
    factory = sdk2.parameters.Bool('If this image contains factory reset partition')
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    hw_test_mode = sdk2.parameters.Bool(
        'If this image is for hardware-testing ONLY (NOT SUITABLE FOR PRODUCTION DEVICES), see QUASAR-7125')


# Platform image unpacked
class QuasarYandexstation2ImageUnpacked(QuasarResourceBase):
    """
    A tarball with unpacked image of Yandex.Station2.
    """


class QuasarYandexstation2ImageBitmaps(QuasarResourceBase):
    """
    A tarball with bitmaps for Yandex.Station2.
    """


# OTA image.
class QuasarYandexstation2OTAImage(_QuasarOTABase):
    """
    An OTA-ready package for Yandex.Station2
    """
    platform = sdk2.parameters.String('Quasar platform',
                                      default=Platform.YANDEXSTATION_2)
    factory = sdk2.parameters.Bool('If this image contains factory reset partition')
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')


class QuasarYandexstation2TargetFiles(_QuasarImageBase):
    """
    A `target_files.zip` file from AOSP build process -- used in later stages, for Yandex.Station2
    """
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    repository_tag = sdk2.parameters.String('Repository tag')


class QuasarYandexstation2LinuxImage(QuasarResourceBase):
    """
    A buildroot-based Linux image for yandexstation_2 sideload in factory mode
    """
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')


class QuasarYandexstation2LinuxImageFP(QuasarYandexstation2LinuxImage):
    """
    A buildroot-based Linux image for yandexstation_2 sideload in factory mode
    """
    pass


class QuasarYandexstation2LinuxImageJIG(QuasarYandexstation2LinuxImage):
    """
    A buildroot-based Linux image for yandexstation_2 sideload in factory mode
    """
    pass


class QuasarChironLinuxImage(QuasarResourceBase):
    """
    A buildroot-based Linux image for chiron sideload in factory mode

    See https://wiki.yandex-team.ru/sandbox/cookbook/#sborkakontejjnerasokruzheniemvkotorombudetispolnjatsjazadacha
    """
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')


class QuasarYandexstation2BootloaderImage(QuasarResourceBase):
    """
    A minimal image for Yandex.Station2 flashing in factory mode
    """
    secureboot = sdk2.parameters.Bool('If this image is secureboot-locked for production')
    version = sdk2.parameters.String('Full version of the bootloader')


class QuasarYandexstation2BootloaderImageFP(QuasarYandexstation2BootloaderImage):
    """
    A minimal image for Yandex.Station2 flashing in factory mode
    """
    pass


class QuasarYandexstation2BootloaderImageJIG(QuasarYandexstation2BootloaderImage):
    """
    A minimal image for Yandex.Station2 flashing in factory mode
    """
    pass


class QuasarYandexstation2OtaTools(QuasarResourceBase):
    """
    Android OTA tools package
    """


class QuasarLinuxImage(QuasarResourceBase):
    """
    An image with linux-only partitions -- suitable for testing and such
    """


class QuasarApp(QuasarResourceBase):
    """
    .apk of the quasar-app
    """
    is_perf = sdk2.parameters.Bool('Is this a perf variant of quasar apk build', default=False)


class QuasarStub(QuasarResourceBase):
    """
    .apk of the quasar-stub (a-la Bolvanka)
    """


class QuasarJniApp(QuasarResourceBase):
    """
    .apk of the yandex_io Jni Example Apk
    """


class QuasarYandexTvApp(QuasarResourceBase):
    """
    .apk from smart_devices/platforms/yandex_tv/apk
    """


class QuasarCentaurDemoApp(QuasarResourceBase):
    """
    demo-.apk from smart_devices/platforms/centaur
    """


class QuasarCentaurUpdaterApp(QuasarResourceBase):
    """
    updater.apk for centaur platform (from :tv:updater-app)
    """


class QuasarCentaurApp(QuasarResourceBase):
    """
    launcher-.apk from smart_devices/platforms/centaur
    """


class QuasarServices(QuasarResourceBase):
    """
    .apk of the quasar-services
    """


class AudioSenderApp(QuasarResourceBase):
    """
    .apk of the speechkit audio-sender
    """


class QuasarDaemons(QuasarResourceBase):
    """
    Directory with quasar binary deamons
    """
    quasar_platform = sdk2.parameters.String('Platform these binaries were built for.')
    quasar_platform_revision = sdk2.parameters.String('Platform revision.')
    commit_time = sdk2.parameters.String('Time of commit to arcadia this binaries were built from.')
    svn_revision = sdk2.parameters.String('Svn revision this binaries were built from.')

    major_release_num = sdk2.parameters.Integer('Major release number', default=None)
    minor_release_num = sdk2.parameters.Integer('Minor release number', default=None)

    def is_before(self, release, trunk_revision):
        """
        Returns True, if deamons were built from trunk revision prior to [trunk_revision]
        or from any release branch prior to [release].
        """
        return (
            (self.major_release_num is not None and int(self.major_release_num) < release) or
            int(self.svn_revision) < trunk_revision
        )


class QuasarFactoryDaemons(QuasarDaemons):
    """
    Directory with quasar binary factory deamons
    """


class QuasarSymbols(QuasarResourceBase):
    """
    Directory with symbols for quasar daemons
    """
    quasar_platform = sdk2.parameters.String('Device for this image')


class QuasarMiniDaemons(QuasarDaemons):
    pass


class QuasarMiniSymbols(QuasarSymbols):
    pass


class QuasarBuildrootPkgCache(sdk2.Resource):
    ttl = "inf"
    auto_backup = True
    version = sdk2.parameters.String('Version of the cache')

    quasar_platform = sdk2.parameters.String('Target platform of Buildroot build system')


class QuasarGstreamer(QuasarResourceBase):
    """
    GStreamer libs for different platforms
    """
    platform = sdk2.parameters.String('Platform for this set of libs')


class PgToYtExecutable(sdk2.Resource):
    auto_backup = True
    executable = True
    releasable = True


class PortoLayerQuasarBackend(QuasarResourceBase):
    pass


class QuasarBackendBinaries(QuasarResourceBase):
    pass


class QuasarBackendConfigs(QuasarResourceBase):
    pass


class QuasarJblLinkMusicImage(_QuasarImageBase):
    pass


class QuasarJblLinkMusicOTAImage(_QuasarOTABase):
    """
    An OTA package for JBL Link Music
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.JBL_LINK_MUSIC)


class QuasarJblLinkPortableImage(_QuasarImageBase):
    pass


class QuasarJblLinkPortableOTAImage(_QuasarOTABase):
    """
    An OTA package for JBL Link Portable
    """
    platform = sdk2.parameters.String('Quasar platform', default=Platform.JBL_LINK_PORTABLE)


class QuasarJblLinkLxcImage(QuasarLxcImageBase):
    """
    LXC image to run image building tasks for JBL.
    """


class QuasarBackendAmmoFile(sdk2.Resource):
    """
    A tasklet-generated ammo file for quasar backend, based on prod-balancer logs
    """
    ttl = 180
    is_primary_ammo = sdk2.parameters.Bool(default=False)
    releasable = True


class QuasarQuasmodromBinary(QuasarResourceBase):
    pass


class QuasarQuasmodromPackage(QuasarResourceBase):
    pass


class QuasarQuasmodromGroupConfigs(sdk2.Resource):
    ttl = 360
    releasable = True


class QuasarHwtestBinary(sdk2.Resource):
    ttl = 360
    releasable = True


class QuasarJblReleaseNotes(QuasarResourceBase):
    pass


class QuasarJblDebugNotes(QuasarResourceBase):
    pass


class QuasarGeodata(QuasarResourceBase):
    """
    A directory with last released files required for geobase
    """
    releasers = ['QUASAR']


class QuasarTaskletArchive(sdk2.Resource):
    """
    A directory with tasklet executable and (maybe) another executable or data
    """

    releasable = True


class QuasarMetricsTaskletArchive(sdk2.Resource):
    """Tasks release package for quasar/metrics"""

    releasable = True


class QuasarZigbeeFirmware(QuasarResourceBase):
    pass


class YandexIoPackageAnalysisReport(sdk2.Resource):
    """JSON with YandexIO package size analysis report"""


class FactoryResourceBase(sdk2.Resource):
    releasers = ['QUASAR']
    ttl = 30
    releasable = True
    executable = True


class FactoryFlashpointMicroBinary(FactoryResourceBase):
    pass


class FactoryFabricationMicroBinary(FactoryResourceBase):
    pass


class FactoryFlashpointMidiBinary(FactoryResourceBase):
    pass


class FactoryFabricationMidiBinary(FactoryResourceBase):
    pass


class FactoryFlashpointModule2Binary(FactoryResourceBase):
    pass


class FactoryFabricationModule2Binary(FactoryResourceBase):
    pass


class FactoryFlashpointPlutoBinary(FactoryResourceBase):
    pass


class FactoryFabricationPlutoBinary(FactoryResourceBase):
    pass


class FactoryJangles(FactoryResourceBase):
    pass


class FactorySleeve(FactoryResourceBase):
    pass


class FactorySleeveArm(FactoryResourceBase):
    pass


class FactoryElenchusMidiBinary(FactoryResourceBase):
    pass


class FactoryElenchusMiniPackage(FactoryResourceBase):
    pass


class FactoryElenchusMiniBinary(FactoryResourceBase):
    pass
