import logging

import sandbox.common.types.task as ctt
from sandbox.common.types.misc import DnsType

from sandbox.common.utils import get_task_link

from sandbox import sdk2
from sandbox.sdk2.vcs import svn

from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.quasar.station_factory.image_build import QuasarBuildImageFactory
from sandbox.projects.quasar.station_factory.services_build import QuasarBuildServicesFactory

from sandbox.projects.quasar.resource_types import QuasarApp, QuasarDaemons, QuasarServices
from sandbox.projects.quasar.resource_types.deprecated import QuasarOTAImage, QuasarDeviceImage
from sandbox.projects.quasar.build_types import ImageBuildtype
from sandbox.projects.quasar.station_factory.platform import Platform
from sandbox.projects.quasar.utils import Slack, VCSSelector, VCS


def one(finding):
    """
    helper for sandbox lookup APIs
    Use like::
        a_task = one(sdk2.Task.find(id=some_id))
    """
    return next(iter(finding.limit(1)))


class QuasarBuildFullFactory(sdk2.Task):
    """
    Builds services and image with those services via calling other tasks
    """
    class Parameters(sdk2.Task.Parameters):
        vcs = VCSSelector("VCS")

        kill_timeout = 5 * 3600  # can take long, duh

        tag = sdk2.parameters.String('Git tag or branch for platform repo', default='develop', required=True)

        with vcs.value[VCS.GIT]:
            services_tag = sdk2.parameters.String('Git tag or branch for services repo', default='develop', required=True)

        with vcs.value[VCS.SVN]:
            arcadia_services_url = sdk2.parameters.ArcadiaUrl(
                "Arcadia URL", default_value=svn.Arcadia.ARCADIA_TRUNK_URL
            )
            arcadia_patch = sdk2.parameters.String(
                build_parameters.ArcadiaPatch.description,
                default=build_parameters.ArcadiaPatch.default,
                multiline=True
            )

        release = sdk2.parameters.Bool('Release built tasks', default=False)
        slack_tag = sdk2.parameters.String('Slack notification message -- leave empty to not notify', default='')

        with sdk2.parameters.Group('Internal'):
            test = sdk2.parameters.Bool('Test any non-prod functionality in build process', default=False)

    class Requirements(sdk2.Task.Requirements):
        cores = 1  # we are light task, only call other tasks
        dns = DnsType.DNS64  # for slack API

        class Caches(sdk2.Requirements.Caches):
            pass

    def notify(self, text, attachments=[]):
        """
        Possibly notifies with given `text` and `attachments`
        """
        if self.Parameters.slack_tag:
            task = Slack.attachment(title="Task #" + str(self.id), title_link=get_task_link(self.id))

            Slack.notify(
                text="%s[%s]: %s" % (self.type, self.Parameters.slack_tag, text),
                attachments=[task] + attachments)

    def on_execute(self):
        try:
            self.do_execute()
        except Exception:
            self.notify('build failed with exception')
            raise

    def do_execute(self):
        with self.memoize_stage.build_services:
            self.notify("build started")

            logging.info('Building services...')
            description = 'Building services for task {} {} {}'.format(
                self.id,
                "tag" if self.Parameters.vcs == VCS.GIT else "path",
                self.Parameters.tag if self.Parameters.vcs == VCS.GIT else self.Parameters.arcadia_services_url
            )
            build_services = QuasarBuildServicesFactory(
                self,
                description=description,
                notifications=self.Parameters.notifications,
                priority=self.Parameters.priority,
            )

            self.Context.build_services_task_id = build_services.id

            build_services.Parameters.report_github_statuses = False
            build_services.Parameters.branch = self.Parameters.services_tag
            build_services.Parameters.vcs = self.Parameters.vcs
            build_services.Parameters.arcadia_url = self.Parameters.arcadia_services_url
            build_services.Parameters.arcadia_patch = self.Parameters.arcadia_patch
            build_services.Parameters.publish_resources = True

            build_services.save()
            build_services.enqueue()

            raise sdk2.WaitTask([build_services], ctt.Status.Group.FINISH, wait_all=True, timeout=3600)  # 60 min max

        with self.memoize_stage.build_image:
            logging.info('Building image...')

            build_services_task = one(sdk2.Task.find(id=self.Context.build_services_task_id, children=True))

            logging.info('Located services task as %s' % build_services_task)

            def build_image(**params):
                task = QuasarBuildImageFactory(
                    self,
                    description='Building image for task %s tag %s' % (self.id, self.Parameters.tag),
                    notifications=self.Parameters.notifications,
                    priority=self.Parameters.priority,
                )

                task.Parameters.tag = self.Parameters.tag
                task.Parameters.publish_to_s3 = True  # always release full build's OTA to S3
                task.Parameters.quasar_app = one(QuasarApp.find(task=build_services_task))
                task.Parameters.quasar_services = one(QuasarServices.find(task=build_services_task))
                task.Parameters.quasar_daemons = one(QuasarDaemons.find(task=build_services_task, quasar_platform=Platform.yandexstation))

                for (name, value) in params.items():
                    setattr(task.Parameters, name, value)

                task.save()
                task.enqueue()

                return task

            tasks = {
                build_type: build_image(android_build_type=build_type, sign=sign)
                for (build_type, sign) in [(ImageBuildtype.ENGINEERING, False), (ImageBuildtype.USER, True)]
            }

            self.Context.build_image_tasks = {bt: task.id for (bt, task) in tasks.items()}

            raise sdk2.WaitTask(tasks.values(), ctt.Status.Group.FINISH, wait_all=True, timeout=3600 * 5)  # 5 hours max

        if self.Parameters.release:
            release_props = dict(type=ctt.ReleaseStatus.STABLE, subject='Full build for tag %s' % self.Parameters.tag)
            tasks_to_release = self.Context.build_image_tasks.values() + [self.Context.build_services_task_id]

            for task in tasks_to_release:
                logging.info('Releasing task %s with props %s' % (task, release_props))

                self.server.release(task_id=task, **release_props)

                logging.info('Released')

    def on_failure(self, prev_status):
        self.notify('build failed')

    def on_success(self, prev_status):
        def res(build_type, res_type, **attrs):
            """
            Convenience helper to get resource object from `build_image_task` given its type and attributes
            """

            return one(res_type.find(task_id=self.Context.build_image_tasks[build_type], attrs=dict(attrs, buildtype=build_type)))

        PURPLE = '990099'
        GREEN = '00FF00'
        BLACK = '000000'

        self.notify(
            "build finished",
            [
                # version is same for all artifacts, take this
                Slack.attachment("Version is %s" % res(ImageBuildtype.USER, QuasarOTAImage).version, color=GREEN),

                Slack.attachment(
                    "USER Signed OTA",
                    title_link=res(ImageBuildtype.USER, QuasarOTAImage, signed=True).s3_url, color=PURPLE),
                Slack.attachment(
                    "USER Signed Image",
                    title_link=res(ImageBuildtype.USER, QuasarDeviceImage, signed=True).http_proxy, color=PURPLE),
                Slack.attachment(
                    "ENG OTA",
                    title_link=res(ImageBuildtype.ENGINEERING, QuasarOTAImage, signed=False).s3_url, color=BLACK),
                Slack.attachment(
                    "ENG Image",
                    title_link=res(ImageBuildtype.ENGINEERING, QuasarDeviceImage, signed=False).http_proxy, color=BLACK),
            ],
        )
