import os
import zlib
import logging

from glob import glob
from uuid import uuid4
from datetime import date, datetime

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt

from sandbox.common.types.misc import DnsType
from sandbox.common.errors import TemporaryError

from sandbox import sdk2
from sandbox.sandboxsdk import ssh

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.process import run_process, SandboxSubprocessError
from sandbox.sandboxsdk.paths import copy_path, remove_path

from sandbox.projects.quasar.resource_types import QuasarApp, QuasarServices, \
    QuasarLinuxImage, QuasarDaemons, QuasarImageAndroidCcacheTarball
from sandbox.projects.quasar.resource_types.deprecated import QuasarOTAImage, QuasarDeviceImage, QuasarImageTargetFiles
from sandbox.projects.quasar.station_factory.platform import Platform
from sandbox.projects.quasar.build_types import ImageBuildtype
from sandbox.projects.quasar.utils import LastStableResource, LastResource, SafePublishingMixing, SignerMixin, LastResourceWithAttrs


class QuasarBuildImageFactory(SignerMixin, SafePublishingMixing, sdk2.Task):
    """
    Build full image for quasar device and publish it as a resource
    Optionally, sign it via Signer and publish it to S3
    """
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 5 * 3600  # can take long, duh

        tag = sdk2.parameters.String('Git tag', default='factory', required=True)

        publish_to_s3 = sdk2.parameters.Bool('Publish OTA to S3', default=False)
        sign = sdk2.parameters.Bool('Build signed image and OTA', default=False)
        quasar_app = LastStableResource('Quasar App .apk', resource_type=QuasarApp)
        quasar_services = LastStableResource('Quasar Services .apk', resource_type=QuasarServices)
        quasar_daemons = LastResourceWithAttrs('Quasar binary daemons', resource_type=QuasarDaemons, attrs=dict(released=ctt.ReleaseStatus.STABLE, quasar_platform=Platform.yandexstation))

        with sdk2.parameters.String('Build type', required=True, default=ImageBuildtype.ENGINEERING) as android_build_type:
            for build_type in list(ImageBuildtype):
                android_build_type.values[build_type] = build_type

        with sdk2.parameters.Group('Internal'):
            test = sdk2.parameters.Bool('Test some new and exciting feature', default=False)
            c_cache_tarball = LastResource('Quasar android ccache dir tarball', resource_type=QuasarImageAndroidCcacheTarball)

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64  # for external interactions
        privileged = True  # to run apt-get installs in build-all script
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.SSD  # repo is big
        disk_space = 150 * 1024  # 150 Gb

        environments = [
            PipEnvironment('boto3', use_wheel=True),
        ] + SignerMixin.environments

    GIT_URL = 'ssh://git@bb.yandex-team.ru/quas/r18stationandroid.git'
    VAULT_OWNER = 'QUASAR'
    SSH_PRIVATE_KEY_VAULT_NAME = 'robot-quasar-ssh-private-key'
    CHECKOUT_PATH = 'quasar'

    CCACHE_DIR = '__ccache'  # we always use this as CCACHE_DIR
    CCACHE_SIZE = '20G'

    S3_SECRET_KEY_VAULT_NAME = 'robot-quasar-s3-secret-key'
    S3_SECRET_KEY_ID_VAULT_NAME = 'robot-quasar-s3-secret-key-id'

    def _checkout(self):
        logging.info('Installing git-lfs from github. FIXME: use env, luke!')

        run_process(
            # FIXME: that's not the way we want it to be
            ['wget', 'https://github.com/git-lfs/git-lfs/releases/download/v2.3.4/git-lfs-linux-amd64-2.3.4.tar.gz'],  # noqa
            log_prefix='git_lfs_install')

        run_process(
            # FIXME: that's not the way we want it to be
            ['tar', 'xvzf', 'git-lfs-linux-amd64-2.3.4.tar.gz'],
            log_prefix='git_lfs_install')

        run_process(
            # FIXME: that's not the way we want it to be
            ['bash', 'git-lfs-2.3.4/install.sh'],
            log_prefix='git_lfs_install')

        # uninstall lfs hooks
        # see https://github.com/git-lfs/git-lfs/issues/931
        run_process(
            ['git', 'lfs', 'uninstall'],
            log_prefix='git_lfs_install')

        logging.info('Checkout code for tag <%s> ...' % self.Parameters.tag)

        with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
            # shallow clone to reduce data transfer
            run_process(
                ['git', 'clone',
                 '--depth', '1',
                 '--branch', self.Parameters.tag,
                 self.GIT_URL, self.CHECKOUT_PATH],
                log_prefix='git_clone',
                shell=True,
            )

            # pull lfs files explicitly
            run_process(
                ['git', 'lfs', 'pull'],
                work_dir=self.CHECKOUT_PATH,
                log_prefix='git_lfs_pull',
            )

    def _phase(self, name, phase_arguments=[], resources={}, resources_attrs={}, environment={}):
        """
        A build phase matching one from the `build_all_ci.sh` script
        :param str name: a phase name
        :param list[str] phase_arguments: extra arguments to pass to build script
        :param dict resources: a {<resource_class> : <resource_path>}
            map for resources to be published after phase builds
            <resource_path> should be relative to the self.CHECKOUT_PATH or absolute path
            <resource_path> is glob-expanded via `glob.glob`
        :param dict resources_attrs: a {name: value} dict with extra attributes to be added to all resources
        :param dict environment: extra env to be passed to the `build_all_ci.sh` script

        :return: map of published resources {<resource_class>: <resource_object>}
        """

        logging.info('Running the %s %s phase...' % (name, ''.join(phase_arguments)))

        env = os.environ.copy()
        env.update(environment)

        run_process(
            ['bash',
             os.path.join(self.CHECKOUT_PATH, 'build_all_ci.sh'),
             name] + phase_arguments,
            log_prefix=name,
            environment=env,
        )

        def try_expand(a_path):
            """
            Expands given `a_path` to a first really exising file.
            If no file is found `a_path` is returned as-is.
            """
            expanded = glob(a_path)

            if expanded:
                return expanded[0]
            else:
                logging.warn('No files match %s' % a_path)
                # show unexpanded path so publishing fails
                return a_path

        published = self.publish_safely(
            resources={
                c: try_expand(p if os.path.isabs(p) else os.path.join(self.CHECKOUT_PATH, p))
                for (c, p) in resources.items()
            },
            comment='For branch {}'.format(self.Parameters.tag),
        )

        for resource in published.values():
            map(lambda item: setattr(resource, *item), resources_attrs.iteritems())

        return published

    def _place_resources(self):
        """
        Places resources from parameters (like built .apk files or binary services)
        into source tree so they are picked up by the image build process
        """

        # location is relatively to the repo root
        resource_to_location = {
            self.Parameters.quasar_app: 'R18_android/android/device/softwinner/tulip-d1/packages/Quasar-Webmusic/webmusic.apk',
            self.Parameters.quasar_services: 'R18_android/android/device/softwinner/tulip-d1/packages/Quasar-Services/quasar-services.apk',
            self.Parameters.quasar_daemons: 'R18_android/android/device/softwinner/tulip-d1/quasar/',
        }

        for resource, location in resource_to_location.items():
            if not resource:
                logging.warn('Skipping location <%s> as resource is not given' % location)
                continue

            download_path = str(sdk2.ResourceData(resource).path)

            logging.info('Downloaded res %s to %s' % (resource, download_path))

            dest_path = os.path.join(self.CHECKOUT_PATH, location)
            remove_path(dest_path)
            copy_path(download_path, dest_path)

    def _strip_binaries(self):
        """
        Strips placed binaries of debug symbols.
        Uses pre-built `strip` bundled in android source tree.
        Ignore errors as some files are non-strippable, but we dont know which ones.
        """
        try:
            # TODO: when refactoring, unify paths with `_place_resources`
            run_process(
                '%s * libs/*' % self.path(self.CHECKOUT_PATH, 'R18_android/android/prebuilts/gcc/linux-x86/aarch64/aarch64-linux-android-4.9/aarch64-linux-android/bin/strip'),
                work_dir=str(self.path(self.CHECKOUT_PATH, 'R18_android/android/device/softwinner/tulip-d1/quasar')),
                shell=True,
                log_prefix='strip_daemons',
            )
        except SandboxSubprocessError:
            pass

    def _ccache_command(self, args, log_prefix='ccache'):
        # TODO: we are using this ccache for now, may need tweaking
        ccache = 'R18_android/android/prebuilts/misc/linux-x86/ccache/ccache'

        env = os.environ.copy()
        env['CCACHE_DIR'] = str(self.path(self.CCACHE_DIR))

        return run_process(
            [str(self.path(self.CHECKOUT_PATH, ccache))] + list(args),
            environment=env,
            log_prefix=log_prefix)

    def _ccache_stats(self):
        """
        Calls ccache prebuilt binary to show stats
        """
        return self._ccache_command(['-s'], log_prefix='ccache_stats')

    def _prepare_ccache(self):
        self.untarball_dir_res(self.Parameters.c_cache_tarball, self.path(self.CCACHE_DIR))

        self._ccache_command(['-M', self.CCACHE_SIZE], log_prefix='ccache_set_max_size')

        self._ccache_stats()

    def _publish_ccache(self):
        self._ccache_stats()

        self.tarball_publish_dir(QuasarImageAndroidCcacheTarball, self.path(self.CCACHE_DIR), comment='ccache for android')

    def _determine_version(self):
        """
        :returns: a `str` version for this build.

        Version is defined as `a_1.q_1.a_2.q_2.task_id.build_date`, where:
            * android repo version is `a_1.a_2`
            * quasar repo version is `q_1.q_2`
            * `task_id` is id of current task
            * `build_date` is date in format `20180305`
        """
        def _read_version_file(path):
            """
            :param str path: to version file relatively to the repo root
            :returns: tuple with version parts
            """
            parts = self.path(self.CHECKOUT_PATH, path).read_text().strip().split('.')

            if len(parts) != 2:
                raise ValueError('File %s has invalid version (not two dot-separated parts) of %s' % (path, parts))

            return parts

        a_version = _read_version_file('VERSION')
        q_version = _read_version_file('R18_android/android/device/softwinner/tulip-d1/quasar/VERSION')

        suffix = [str(self.id), date.today().strftime('%Y%m%d')]

        # For factory firmware we always append "factory" regardless of branch name

        suffix.append("factory")

        if self.Parameters.android_build_type != ImageBuildtype.USER and self.Parameters.android_build_type != ImageBuildtype.USERDEBUG:
            suffix.append(str(self.Parameters.android_build_type).upper())

        version = '.'.join(map('.'.join, zip(a_version, q_version)) + suffix)
        logging.info('determined version to be %s' % version)

        return version

    def _place_version(self, version):
        """
        :param str version: a version to replace

        Replaces version placeholder in certain subtrees of sources
        """
        replaced_dirs = ['R18_android/android/device/softwinner/tulip-d1/']

        for replaced_dir in replaced_dirs:
            run_process(
                "grep -rl __QUASAR_VERSION_PLACEHOLDER__ . | xargs sed -i 's^__QUASAR_VERSION_PLACEHOLDER__^%s^g'" % version,
                work_dir=str(self.path(self.CHECKOUT_PATH, replaced_dir)),
                shell=True,
                log_prefix='replace_version',
            )

    def s3_connect(self):
        """
        Reads parameters and configures `self.s3_transfer` (a boto3's `S3Transfer` instance) to work with target bucket

        Ripped of `StrmVideoConvert`.

        TODO: extract a trait. Relies on PipEnvironment for boto3!
        """
        self.s3_write_endpoint_url = "https://s3.mds.yandex.net"
        self.s3_bucket = 'quasar'  # maybe not hardcode?
        self.s3_read_endpoint_url = "https://%s.s3.yandex.net" % self.s3_bucket  # NB: bucket should be publically accessible

        s3_secret_key = sdk2.Vault.data(self.VAULT_OWNER, self.S3_SECRET_KEY_VAULT_NAME)
        s3_secret_key_id = sdk2.Vault.data(self.VAULT_OWNER, self.S3_SECRET_KEY_ID_VAULT_NAME)

        import boto3

        from boto3.s3.transfer import S3Transfer
        from botocore.exceptions import ClientError

        s3_conn = boto3.client(
            's3',
            endpoint_url=self.s3_write_endpoint_url,
            aws_access_key_id=s3_secret_key_id,
            aws_secret_access_key=s3_secret_key)

        try:
            s3_conn.head_bucket(Bucket=self.s3_bucket)
        except ClientError as e:
            error_code = int(e.response['Error']['Code'])
            if error_code == 404:
                s3_conn.create_bucket(Bucket=self.s3_bucket)
            else:
                msg = 's3 get bucket code {}'.format(error_code)
                raise TemporaryError(msg)

        self.s3_transfer = S3Transfer(s3_conn)

        logging.info("S3 connected")

    def s3_upload(self, file_path, target_path, metadata={}, content_type='application/octet-stream'):
        """
        Uploads a single file to the s3 bucket. Connects if not connected.

        :param str file_path: path to an existing file to be uploaded
        :param str target_path: path inside our bucket to upload file to
        :param dict metadata: s3 metadata to store on the file
        :param str content_type: to be set on the file

        :returns: external URL for file access
        """
        if not hasattr(self, 's3_transfer'):
            self.s3_connect()

        self.s3_transfer.upload_file(
            file_path,
            self.s3_bucket,
            target_path,
            extra_args={
                "Metadata": metadata,
                "ContentType": content_type,
            },
        )

        return '/'.join([self.s3_read_endpoint_url.rstrip('/'), target_path.lstrip('/')])

    def s3publish_ota(self, ota_resource):
        """
        Uploads an `QuasarOTAImage` resource to s3 and updates it's properties accordingly
        """

        logging.info("Uploading ota to s3...")

        ota_path = str(sdk2.ResourceData(ota_resource).path)

        def my_crc32(data):
            """
            simulates cmdline tool crc32 -- returns hex of uint32 of crc32-checksum with leading 0x removed
            """
            crc32 = zlib.crc32(data)

            # zlib's crc32 removes int, convert to uint
            if crc32 < 0:
                crc32 = 2 ** 32 + crc32

            return hex(crc32)[2:]  # trim leading 0x

        ota_resource.crc32_checksum = my_crc32(open(ota_path, 'rb').read())

        ota_resource.s3_url = self.s3_upload(
            ota_path,
            'station/ota/%s/%s/quasar-%s.zip' % (ota_resource.buildtype, uuid4(), ota_resource.version),
            metadata={
                'crc32': str(ota_resource.crc32_checksum),
                'updated': datetime.now().isoformat(),
                'version': str(ota_resource.version),
                'buildtype': str(ota_resource.buildtype),
            },
            content_type='application/zip',
        )

        logging.info("Uploaded ota zip to %s, crc32 is %s" % (ota_resource.s3_url, ota_resource.crc32_checksum))

    def sign_images(self, target_files_resource):
        """
        See https://st.yandex-team.ru/SIGNER-173 -- signed archive always contains two files with predefined names

        :param target_files_resource: to sign
        :returns: pair `(signed_target_files_path, signed_ota_path)`
        """
        signed_bundle = self.sign(sdk2.ResourceData(target_files_resource).path, SignerMixin.Certs.TARGET_FILES)

        logging.info("Signed target files to %s" % (signed_bundle))

        extracted_path = str(self.path('_signed_target_files'))
        os.makedirs(extracted_path)

        run_process(
            ['unzip', signed_bundle, '-d', extracted_path],
            log_prefix='extract_bundle',
        )

        return [os.path.join(extracted_path, element) for element in ('target_files.zip', 'ota.zip')]

    def on_execute(self):
        self._checkout()

        self._place_resources()

        # TODO: do it only for USER versions of image
        self._strip_binaries()

        version = self._determine_version()
        self._place_version(version)

        self._prepare_ccache()

        extra_env = {
            # use bundled android ccache
            # see https://source.android.com/setup/initializing#setting-up-ccache
            'USE_CCACHE': '1',
            # use tina linux's ccache build option
            'CONFIG_CCACHE': '1',
            'CCACHE_DIR': str(self.path(self.CCACHE_DIR)),
        }

        self._phase('deps')
        self._phase(
            'kernel',
            [self.Parameters.android_build_type],  # pass buildtype to build kernel properly -- see QUASAR-1714
        )

        self._phase(
            'linux',
            resources={QuasarLinuxImage: 'r18-linux/out/tulip-d1/tina_tulip-d1_uart0.img'},
            # FIXME: linux wont build with ccache now for some reaseon
            # environment=extra_env
        )

        unsigned_artifacts = self._phase(
            'android',
            [self.Parameters.android_build_type],
            resources={
                QuasarDeviceImage: 'R18_android/lichee/tools/pack/sun50iw1p1_android_d1_uart0.img',
                # file name is really different each time
                QuasarImageTargetFiles: 'R18_android/android/out/target/product/tulip-d1/obj/PACKAGING/target_files_intermediates/tulip_d1-target_files-*.zip',
                QuasarOTAImage: 'R18_android/android/out/target/product/tulip-d1/tulip_d1-ota-*.zip'},
            resources_attrs=dict(
                buildtype=self.Parameters.android_build_type,
                version=version,
                signed=False),
            environment=extra_env)

        if self.Parameters.publish_to_s3:
            self.s3publish_ota(unsigned_artifacts[QuasarOTAImage])

        if self.Parameters.sign:
            signed_targetfiles, signed_ota = self.sign_images(unsigned_artifacts[QuasarImageTargetFiles])

            signed_artifacts = self._phase(
                'signed_image',
                [self.Parameters.android_build_type, signed_targetfiles],
                resources={
                    QuasarDeviceImage: 'R18_android/lichee/tools/pack/sun50iw1p1_android_d1_uart0.img',
                    QuasarOTAImage: signed_ota,  # take ota as was signed in signer
                },
                resources_attrs=dict(
                    buildtype=self.Parameters.android_build_type,
                    version=version,
                    signed=True),
                environment=extra_env,
            )

            if self.Parameters.publish_to_s3:
                self.s3publish_ota(signed_artifacts[QuasarOTAImage])

        self._ccache_stats()

        self._publish_ccache()
