import os
import errno
import shutil
import logging
import subprocess

from multiprocessing import cpu_count

from sandbox import sdk2
from sandbox.sdk2.vcs import svn

from sandbox.sandboxsdk import ssh

from sandbox.sdk2.vcs.git import Git

import sandbox.common.types.client as ctc

from sandbox.common.types.misc import DnsType
from sandbox.common.utils import singleton_property

from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.sandbox_ci import managers
from sandbox.projects.sandbox_ci.utils.github import GitHubStatus

from sandbox.projects.quasar.resource_types import QuasarApp, QuasarGradleCacheTarball, QuasarDaemons, QuasarServices
from sandbox.projects.quasar.station_factory.platform import Platform
from sandbox.projects.quasar.utils import SafePublishingMixing, LastResource, RunHelper, SignerMixin, VCSSelector, VCS


class QuasarBuildServicesFactory(SignerMixin, SafePublishingMixing, sdk2.Task):
    """
    Build quasar services.

    NB: this task relies on LXC image for build dependencies
    """

    GIT_URL = 'git@github.yandex-team.ru:quasar-dev/quasar.git'
    VAULT_OWNER = 'QUASAR'
    SSH_PRIVATE_KEY_VAULT_NAME = 'robot-quasar-ssh-private-key'
    ARCADIA_DIR = 'arcadia'
    CHECKOUT_PATH = os.path.join(ARCADIA_DIR, 'quasar', 'yandex_io')
    DEFAULT_CERT = 1  # Android production certificate id from signer

    class Parameters(sdk2.Task.Parameters):
        test = sdk2.parameters.Bool('Test', default=False)
        factory = sdk2.parameters.Bool('Factory. FIXME: REMOVE ME', default=True)
        old_signer = sdk2.parameters.Bool('Old signer', default=True)

        with sdk2.parameters.RadioGroup("Java build type") as java_build_type:
            java_build_type.values['Release'] = java_build_type.Value(value='Release')
            java_build_type.values['Debug'] = java_build_type.Value(value='Debug', default=True)

        run_tests_debug = sdk2.parameters.Bool('Run quasar_tests for Debug', default=False)

        vcs = VCSSelector("VCS")

        with vcs.value[VCS.GIT]:
            branch = sdk2.parameters.String('Git tag', default='factory', required=True)

        with vcs.value[VCS.SVN]:
            arcadia_url = sdk2.parameters.ArcadiaUrl(
                "Arcadia URL", default_value=svn.Arcadia.ARCADIA_TRUNK_URL
            )
            arcadia_patch = sdk2.parameters.String(
                build_parameters.ArcadiaPatch.description,
                default=build_parameters.ArcadiaPatch.default,
                multiline=True
            )

        publish_resources = sdk2.parameters.Bool('Publish resources', default=True)

        with sdk2.parameters.Group('Platforms') as block_platforms:
            build_station = sdk2.parameters.Bool('YandexStation', default=True)
            build_linkplay = sdk2.parameters.Bool('Linkplay', default=False)

        # Required for `GitHubStatusesManager`.
        with sdk2.parameters.Group('GitHub') as block_github:
            report_github_statuses = sdk2.parameters.Bool('Report GitHub statuses', default=True)

            project_github_owner = sdk2.parameters.String('Project owner')
            project_github_repo = sdk2.parameters.String('Project repository')
            project_github_commit = sdk2.parameters.String('Commit hash')
            github_context = sdk2.parameters.String('Github status context')

        _container = sdk2.parameters.Container('Container for Station factory firmware', default_value=506070818)  # container for station factory firmware
        _gradle_cache = LastResource('Quasar android gradle cache', resource_type=QuasarGradleCacheTarball)

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64  # for external interactions
        privileged = True  # to run apt-get installs in build-all script
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.SSD
        disk_space = 20 * 1024  # 20 Gb
        environments = [SandboxJavaJdkEnvironment('1.8.0')] + SignerMixin.environments

    def on_prepare(self):
        """
        As we are privileged=True we need to init repos and caches here
        """

        self.github_statuses.report_self_status(description='Preparing')

        try:
            os.makedirs(self.CHECKOUT_PATH)
        except OSError as exc:
            if exc.errno != errno.EEXIST:
                raise

        if self.Parameters.vcs == VCS.GIT:
            with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
                Git(self.GIT_URL).clone(
                    self.CHECKOUT_PATH, self.Parameters.branch,
                    commit=self.Parameters.project_github_commit
                )

        elif self.Parameters.vcs == VCS.SVN:
            # XXX: unlike in QuasarBuildServices, this code is written with assumption that
            # XXX: a direct path to quasar directory is passed, like arcadia:/arc/branches/quasar/old/factory
            url = self.Parameters.arcadia_url
            svn.Arcadia.export(url, self.CHECKOUT_PATH)
            if self.Parameters.arcadia_patch:
                sdk2.svn.Arcadia.apply_patch(self.ARCADIA_DIR, self.Parameters.arcadia_patch, self.path())

        else:
            raise ValueError('Supported VCS are limited to Git and SVN so far')

    @property
    def env(self):
        """
        Read env from what was written to /etc/profile in LXC container preparation
        """
        try:
            logging.info('env is %r' % self._env)
            return self._env
        except AttributeError:
            self._env = dict(l.split('=', 1) for l in
                             subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
        logging.info('env is %r' % self._env)
        return self._env

    def on_execute(self):
        logging.info('Starting execution...')

        self.github_statuses.report_self_status(description='Running')

        self._prepare_gradle_cache()

        signer_oauth_token = sdk2.Vault.data('robot-quasar-signer-oauth-token')

        env = os.environ.copy()
        env.update(self.env)
        env.update(GRADLE_USER_HOME=self.gradle_user_home)
        env.update(YANDEX_SIGNER_OAUTH=signer_oauth_token)

        run = RunHelper(env=env, base_dir=self.CHECKOUT_PATH)

        daemons_platforms = [
            platform for platform, parameter in
            [
                (Platform.yandexstation, self.Parameters.build_station),
                (Platform.linkplay_a98, self.Parameters.build_linkplay),
            ] if parameter
        ]

        if self.Parameters.branch == 'master':
            # FIXME: remove that when develop gets merged
            logging.warn('FIXME: building only for station in master!')
            daemons_platforms = [Platform.yandexstation]

        if self.Parameters.test:
            run("./build_all_ci.sh", "all", "Release", "/toolchains", "", "", self.Parameters.java_build_type)
            for platform in daemons_platforms:
                with run.cd('daemons'):
                    logging.info('Installing platform %s' % platform)
                    with run.env(DESTDIR=str(self.path(self.CHECKOUT_PATH, 'daemons_out', platform))):
                        with run.cd('quasar.%s.Release' % platform):
                            run('make', 'install', '-j', cpu_count())

                        # self.publish(
                        #     # this dir contains all the important outputs
                        #     {QuasarDaemons: 'daemons_out/%s/quasar/' % platform},
                        #     quasar_platform=platform,
                        # )
        elif self.Parameters.factory:
            # FIXME:
            # FIXME: this is a copy of older code for when `factory` was a viable version
            # FIXME: please, get rid of `factory` branch and of this code: see https://st.yandex-team.ru/QUASAR-2064
            # FIXME:
            logging.warn('Building with legacy options for FACTORY: FIXME: see https://st.yandex-team.ru/QUASAR-2064')

            with run.cd('daemons'):
                with run.env(DESTDIR=str(self.path(self.CHECKOUT_PATH, 'daemons_out')), PATH='{NDK_14_TOOLCHAIN}/bin/:{PATH}'.format(**env)):
                    run('./generate.sh', 'x86_64', 'Release')

                    with run.cd('quasar.x86_64.Release'):
                        run('make', '-j', cpu_count())
                        run('./tests/quasar_tests')

                    run('./generate.sh', 'android', 'Release')

                    with run.cd('quasar.android.Release'):
                        run('make', 'install', '-j', cpu_count())

            self.publish(
                # this dir contains all the important outputs
                {QuasarDaemons: 'daemons_out/quasar/'},
                quasar_platform=Platform.yandexstation,
            )
        else:
            with run.cd('daemons'):
                logging.info('Running tests on x86_64')

                run('./generate.sh', 'x86_64', 'Release')

                with run.cd('quasar.x86_64.Release'):
                    run('make', '-j', cpu_count())
                    run('./tests/quasar_tests')

            if self.Parameters.run_tests_debug:
                run('./generate.sh', 'x86_64', 'Debug')

                with run.cd('quasar.x86_64.Debug'):
                    run('make', '-j', cpu_count())
                    run('./tests/quasar_tests')

            platform_extra_env = {
                Platform.yandexstation: dict(PATH='{NDK_14_TOOLCHAIN_64}/bin/:{PATH}'.format(**env)),

                # FIXME: next two are apparently wrong now! Build and add extra SDK!
                Platform.linkplay_a98: dict(PATH='{NDK_14_TOOLCHAIN_32}/bin/:{PATH}'.format(**env)),
            }

            for platform in daemons_platforms:
                logging.info('Building for platform %s' % platform)

                with run.env(DESTDIR=str(self.path(self.CHECKOUT_PATH, 'daemons_out', platform)), **platform_extra_env.get(platform, {})):
                    run('./generate.sh', platform, 'Release')

                    with run.cd('quasar.%s.Release' % platform):
                        run('make', 'install', '-j', cpu_count())

                    self.publish(
                        # this dir contains all the important outputs
                        {QuasarDaemons: 'daemons_out/%s/quasar/' % platform},
                        quasar_platform=platform,
                    )

        with run.cd('protobuf'):
            run('./generate.sh')

        with run.cd('android'):

            run('./gradlew', 'clean', 'ci_test')

            if self.Parameters.test:
                if self.Parameters.old_signer:
                    # TODO https://st.yandex-team.ru/QUASAR-4616: Migrate to new yandex-signer
                    run('./gradlew', 'clean', ':quasar-app:assembleRelease')
                    self.publish({QuasarApp: self.sign(
                        self.path(self.CHECKOUT_PATH, 'android/quasar-app/build/outputs/apk/release/quasar-app-release.apk'),
                        SignerMixin.Certs.APK,
                    )})
                else:
                    run('./gradlew', 'clean', ':quasar-app:assembleRelease', '-Pyandex-signer-oauth=' + signer_oauth_token)
                    self.publish({QuasarApp: 'android/quasar-app/build/outputs/apk/release/quasar-app-release-signed.apk'})
            else:
                # TODO: change from debug to prod once
                if self.Parameters.old_signer:
                    run('./gradlew', 'clean', ':quasar-app:assembleDebug')
                    self.publish({QuasarApp: self.sign(
                        self.path(self.CHECKOUT_PATH, 'android/quasar-app/build/outputs/apk/debug/quasar-app-debug.apk'),
                        SignerMixin.Certs.APK,
                    )})

                    run('./gradlew', 'clean', ':quasar-services:assembleDebug')
                    self.publish({QuasarServices: self.sign(
                        self.path(self.CHECKOUT_PATH, 'android/quasar-services/build/outputs/apk/debug/quasar-services-debug.apk'),
                        SignerMixin.Certs.APK,
                    )})
                else:
                    run('./gradlew', 'clean', ':quasar-app:assembleDebug', '-Pyandex-signer-oauth=' + signer_oauth_token)
                    self.publish({QuasarApp: 'android/quasar-app/build/outputs/apk/debug/quasar-app-debug-signed.apk'})

        self._publish_gradle_cache()

    def publish(self, resources, **resources_attrs):
        """
        Publishes resource.

        To make sure that they are not removed later the resource is first copied
        to some "safe" dir in the task home.

        :param resources:
            dict {resource_class: 'path/to/resorces'} to be published

        :param **resources_attrs:
            extra attributes for resources to be assigned
        """

        if not self.Parameters.publish_resources:
            logging.info('Skipping publish because parameter publish_resources is not True')

            return

        published = self.publish_safely(
            resources={c: p if os.path.isabs(p) else os.path.join(self.CHECKOUT_PATH, p) for (c, p) in resources.items()},
            comment='For branch {}'.format(self.Parameters.branch))

        for resource in published.values():
            map(lambda item: setattr(resource, *item), resources_attrs.iteritems())

        return published

    def on_success(self, prev_status):
        # Should specify state explicitly because of task is in FINISHING status.
        self.github_statuses.report_self_status(state=GitHubStatus.SUCCESS, description='Built')

    def on_failure(self, prev_status):
        # Should specify state explicitly because of task is in FINISHING status.
        self.github_statuses.report_self_status(state=GitHubStatus.FAILURE, description='Failed')

    # here go extra fields for the GitHubStatus manager to work, see it's code

    @singleton_property
    def github_statuses(self):
        # see sandbox.projects.sandbox_ci.github.GitHubStatus
        os.environ['GITHUB_API_TOKEN'] = sdk2.Vault.data('robot-quasar-github-api-token')

        return managers.GitHubStatusesManager(self)

    @property
    def github_context(self):
        # see sandbox.projects.sandbox_ci.github.GitHubStatus
        return self.Parameters.github_context

    @property
    def project_conf(self):
        # see sandbox.projects.sandbox_ci.github.GitHubStatus
        return {'github': {"report_statuses": True}}

    @property
    def gradle_user_home(self):
        return str(self.path('__gradle_user_home'))

    def _prepare_gradle_cache(self):
        self.untarball_dir_res(self.Parameters._gradle_cache, self.gradle_user_home)

    def _publish_gradle_cache(self):
        # cleaning up useless data to prevent cache growing
        # TODO: maybe remove trasforms-X ?
        shutil.rmtree(os.path.join(self.gradle_user_home, 'caches/transforms-1'))

        self.tarball_publish_dir(QuasarGradleCacheTarball, self.gradle_user_home, comment='Gradle home dir for quasar services buildcache')
