import json
import logging
import subprocess

import sandbox.sdk2 as sdk2
import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.errors as exc
import sandbox.projects.common.binary_task as binary_task

from sandbox.projects.mobile_apps.teamcity_sandbox_runner.runner_launcher import TeamcitySandboxRunnerLauncher
from sandbox.projects.quasar.resource_types.tv import AndroidTvApp


class TeamcityApkBuildWrapper(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # vCores
        ram = 1024  # Mb
        disk_space = 10 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(binary_task.LastBinaryReleaseParameters):
        _container = sdk2.parameters.Container("Container with pyaxmlparser", default_value=2362239115)

        branch = sdk2.parameters.String('Branch')
        commit = sdk2.parameters.String('Commit')
        ssh_key = sdk2.parameters.Vault('SSH key')
        config_path = sdk2.parameters.String('Config path')
        resource_attributes = sdk2.parameters.Dict('Resource attributes')
        build_results_root = sdk2.parameters.String('Build results root', default='**/build/outputs/apk')
        extra_artifacts = sdk2.parameters.JSON('Extra artifacts', default={})
        env = sdk2.parameters.Dict('env')

    def run_teamcity_build(self):
        env_branch = '<default>' if self.Parameters.branch == 'trunk' else self.Parameters.branch
        env = {
            'TEAMCITY_BUILD_BRANCH': env_branch,
        }
        if self.Parameters.env:
            env.update(self.Parameters.env)

        subtask = TeamcitySandboxRunnerLauncher(
            self,
            description=self.Parameters.description,
            release_type='stable',
            repo_url='arcadia',
            branch='arcadia/{}'.format(self.Parameters.branch),
            commit=self.Parameters.commit,
            config_from_repository=True,
            config_path=self.Parameters.config_path,
            ssh_key=self.Parameters.ssh_key,
            env=json.dumps(env),
        )

        subtask.save()
        subtask.enqueue()
        self.Context.teamcity_launcher_task_id = subtask.id

    def wait_teamcity_build(self):
        raise sdk2.WaitTask(self.Context.teamcity_launcher_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def check_teamcity_build(self):
        subtask = sdk2.Task[self.Context.teamcity_launcher_task_id]
        if subtask.status not in ctt.Status.Group.SUCCEED:
            raise exc.SandboxTaskFailureError('Subtask failed. More details in subtask {}.'.format(subtask.id))

    def save_to_context(self, key, value):
        current_value = getattr(self.Context, key)
        if current_value:
            raise RuntimeError('Trying to override ctx key {}={} with {}'.format(key, value, current_value))

        setattr(self.Context, key, value)

    def publish_build_results(self):
        task = sdk2.Task[self.Context.teamcity_launcher_task_id]
        artifacts_res = sdk2.Resource.find(task=task, type='TEAMCITY_ARTIFACTS').first()
        artifacts_data = sdk2.ResourceData(artifacts_res)

        for build_results_root in artifacts_data.path.rglob(self.Parameters.build_results_root):
            build_results = build_results_root.rglob('**/*.apk')
            for apk_file in build_results:
                logging.info('Publishing: {}'.format(apk_file))
                build_type = str(apk_file.relative_to(build_results_root).parent).replace('/', '_')
                signed = '-unsigned' not in str(apk_file)

                with sdk2.helpers.ProcessLog(self, logger="apk_parser") as pl:
                    commands = [
                        "from pyaxmlparser import APK",
                        "apk = APK('{}')".format(apk_file),
                        "print(apk.package)",
                        "print(apk.application)",
                        "print(apk.version_code)",
                        "print(apk.version_name)",
                        "print(apk.get_attribute_value('meta-data', 'value', name='yndx.build.generation'))",
                    ]
                    output = subprocess.check_output(['python', '-c', ';'.join(commands)], stderr=pl.stderr)
                outputs = output.split('\n')

                attributes = dict(
                    package_name=outputs[0],
                    app_name=outputs[1],
                    version_code=outputs[2],
                    version_name=outputs[3],
                    generation=outputs[4],
                    signed=signed,
                    build_type=build_type,
                    branch=self.Parameters.branch,
                    commit=self.Parameters.commit,
                )
                attributes.update(self.Parameters.resource_attributes or {})

                resource = AndroidTvApp(self, apk_file.name, apk_file.name)
                for key, value in attributes.items():
                    setattr(resource, key, value)

                resource_data = sdk2.ResourceData(resource)
                sdk2.paths.copy_path(str(apk_file), str(resource_data.path))
                resource_data.ready()

                context_field = attributes['package_name'].replace('.', '_') + '_' + attributes['build_type']
                if not signed:
                    context_field += '_unsigned'

                self.save_to_context(context_field, resource.id)

        for name, artifact in self.Parameters.extra_artifacts.items():
            path = artifact['path']
            resource_type = artifact['resource_type']
            attributes = artifact.get('attributes') or {}

            logging.info('Publishing: {}'.format(path))
            resource = sdk2.Resource[resource_type](self, name, path)

            attributes.update(self.Parameters.resource_attributes or {})
            for key, value in attributes.items():
                setattr(resource, key, value)

            resource_data = sdk2.ResourceData(resource)
            sdk2.paths.copy_path(str(artifacts_data.path / path), str(resource_data.path))
            resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.run_teamcity_build:
            self.run_teamcity_build()

        with self.memoize_stage.wait_teamcity_build:
            self.wait_teamcity_build()

        self.check_teamcity_build()
        self.publish_build_results()
