# -*- coding: utf-8 -*-
import logging
import json
import re
import requests

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel


logger = logging.getLogger(__name__)


class SmartDevicesUpdateRegressTicket(sdk2.Task):
    MAD_HATTER_GROUP = 'WONDERLAND'
    MAD_HATTER_ST_TOKEN = 'env.STARTREK_OAUTH_TOKEN'

    class Parameters(sdk2.Task.Parameters):
        assessors_ticket_key = sdk2.parameters.String('Assessors ticket key', default=None, required=False)
        target_platform = sdk2.parameters.String('Target platform', required=True)
        branch_number = sdk2.parameters.String('Branch', required=True)
        tag_number = sdk2.parameters.String('Tag', required=True)
        firmware_link = sdk2.parameters.String('Firmware link', required=True)
        build_variant = sdk2.parameters.String('Build variant', required=True)

    class Requirements(task_env.StartrekRequirements):
        pass

    def on_execute(self):
        assessors_ticket_key = self.Parameters.assessors_ticket_key
        target_platform = self.Parameters.target_platform
        branch_number = self.Parameters.branch_number
        tag_number = self.Parameters.tag_number
        firmware_link = self.Parameters.firmware_link
        build_variant = self.Parameters.build_variant

        if assessors_ticket_key is None:
            logger.info("Regress ticket is not specified")
            return

        st_token = self.get_tokens()

        from startrek_client import Startrek
        st_client = Startrek(useragent="smart_devices", token=st_token)

        try:
            ticket = st_client.issues[assessors_ticket_key]
            if ticket.status.key != 'open':
                logger.info("Regress ticket isn't in 'open' state. Don't edit it")
                return

            description = ticket.description

            prefix = self.get_build_variant_prefix(build_variant)
            firmware = self.get_firmware_string(branch_number, tag_number, firmware_link, prefix)
            description = self.update_firmware_info(description, firmware, prefix)

            config = self.get_config_info(target_platform)
            description = self.update_config_info(description, config)

            ticket.update(description=description)
        except Exception as error:
            raise Exception("Unable to update regress ticket, reason: {error}".format(error=error))

    @staticmethod
    def get_build_variant_prefix(build_variant):
        prefix = '\n'
        if build_variant is not None:
            prefix += '{}: '.format(build_variant.upper())
        return prefix

    @staticmethod
    def get_firmware_string(branch, tag, firmware, build_variant_prefix):
        return build_variant_prefix + 'stable-{branch}-{tag}: {firmware}\n'.format(
            branch=branch,
            tag=tag,
            firmware=firmware
        )

    @staticmethod
    def update_firmware_info(description, firmware, build_variant_prefix):
        pattern = build_variant_prefix + 'VERSION_PLACEHOLDER'
        if description.find(pattern) != -1:
            logger.info('Found firmware placeholder')
        else:
            logger.info('Firmware placeholder not found')
            pattern = r'{prefix}stable-[^\n]*\n'.format(prefix=build_variant_prefix)
        return re.sub(pattern, firmware, description, re.MULTILINE)

    @staticmethod
    def get_config_info(platform):
        url = 'http://int.quasar.yandex.net/admin/get_system_config?platform={platform}&group=default'.format(
            platform=platform
        )
        response = requests.get(url)
        if response.status_code != 200:
            return '%%(json)cant get config%%\n'
        parsed = json.loads(response.text)
        pretty = json.dumps(parsed, indent=4, sort_keys=True)

        return '%%(json)\n{config}\n%%\n'.format(config=pretty)

    @staticmethod
    def update_config_info(description, config):
        if description.find('Раскрыть конфиг') == -1:
            return description

        pattern = 'CONFIG_PLACEHOLDER'
        if description.find(pattern) != -1:
            logger.info('Found config placeholder')
        else:
            logger.info('Config placeholder not found')
            pattern = r'%%\(json\)\n\{.*\}\n%%\n'
        return re.sub(pattern, config, description, flags=re.DOTALL)

    def get_tokens(self):
        logger.info(
            "Receiving st {st} tokens of group {group}".format(
                st=self.MAD_HATTER_ST_TOKEN,
                group=self.MAD_HATTER_GROUP,
            )
        )

        try:
            st_token = channel.task.get_vault_data(self.MAD_HATTER_GROUP, self.MAD_HATTER_ST_TOKEN)
        except Exception as error:
            raise Exception("Unable to get tokens, reason: {error}".format(error=error))

        logger.info("Tokens received successfully")

        return st_token
