# coding: utf-8

from __future__ import unicode_literals

import base64
import logging

import requests

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.quasar.jangles_lib import LocalJanglesMixin


class UploadJanglesBurnData(LocalJanglesMixin, binary_task.LastBinaryTaskRelease, sdk2.Task):
    # upload to sandbox via ./upload_jangles_burn_data upload --attr 'name=UploadJanglesBurnData' and release resource
    TASKS_RESOURCE_NAME = 'UploadJanglesBurnData'
    upload_endpoint = 'http://localhost:{port}/device_info_package/batch'.format(port=LocalJanglesMixin.LOCAL_PORT)

    # noinspection DuplicatedCode
    class Parameters(binary_task.LastBinaryReleaseParameters):

        with sdk2.parameters.Group('SSH'):
            user = sdk2.parameters.String('SSH user', default='yandex')
            # noinspection SpellCheckingInspection
            host = sdk2.parameters.String('SSH host', default='fenda.ext.eine.yandex.net')
            port = sdk2.parameters.Integer('SSH port', default=1222)

        with sdk2.parameters.Group('Filters'):
            device_model = sdk2.parameters.String('Device model', default='yandex_station_pro')

        with sdk2.parameters.Group('YT'):
            with sdk2.parameters.RadioGroup('YT Cluster') as yt_cluster:
                yt_cluster.values.hahn = yt_cluster.Value(value='hahn', default=True)
                yt_cluster.values.arnold = yt_cluster.Value(value='arnold')

            yt_path = sdk2.parameters.String('Source table', required=True)
            # noinspection SpellCheckingInspection
            yt_secret = sdk2.parameters.YavSecret('YAV secret YT token', default='sec-01d2ffwrdbwyj37zkj4r8zegsn')
            yt_secret_key = sdk2.parameters.String('Key to extract from YAV secret',
                                                   default_value='robot-quasar-yt-token')

        with sdk2.parameters.Group('YAV'):
            # noinspection SpellCheckingInspection
            yav_secret = sdk2.parameters.YavSecret('YAV secret with ssh key', default='sec-01dh4eer2d3pt81r5jfvezqefn')
            yav_secret_key = sdk2.parameters.String('Key to extract from YAV secret', default_value='yandex.key.txt')

    @property
    def binary_executor_query(self):
        return {"attrs": {'released': ctt.ReleaseStatus.STABLE, 'name': self.TASKS_RESOURCE_NAME}}

    def init_yt(self):
        import yt.wrapper as yt

        yt.config.set_proxy(self.Parameters.yt_cluster)
        yt.config['token'] = self.Parameters.yt_secret.data()[self.Parameters.yt_secret_key]

    @classmethod
    def prepare_value(cls, v, b64encode=False):
        if b64encode:
            v = base64.b64encode(v)
        return v.decode()

    @classmethod
    def split_into_parts(cls, data, n):
        start = 0
        while start < len(data):
            yield data[start:start + n]
            start += n

    @classmethod
    def convert_data(cls, row, device_model, process_keyboxes):
        keybox_mapping = (
            (b'wv_keybox_wrapped', 'widevine_keybox'),
            (b'hdcp_22_key_wrapped', 'hdcp_keybox'),
            (b'hdcp_22_key_wrapped', 'hdcp_22_keybox'),
            (b'hdcp_14_key_wrapped', 'hdcp_14_keybox'),
        )

        data = {
            'device_id': cls.prepare_value(row[b'device_id']),
            'color': cls.prepare_value(row[b'color']).upper(),
            'sequence_number': row[b'nmb'],
            'device_model': device_model.value,
        }

        if process_keyboxes:
            for keybox_from, keybox_to in keybox_mapping:
                data[keybox_to] = cls.prepare_value(row[keybox_from], b64encode=True)

        return data

    @classmethod
    def read_data(cls, table):
        import yt.wrapper as yt
        from quasar.manufacturing.jangles.models.device_model import DeviceModel

        keybox_free_models = (
            DeviceModel.YANDEX_MINI_2,
            DeviceModel.YANDEX_MICRO,
            DeviceModel.YANDEX_MODULE_2,
        )

        data = []
        for row in yt.read_table(table, format=yt.YsonFormat(attributes={'format': 'text'}, encoding=None)):
            dt_key = b'device_type'
            if dt_key in row:
                device_model = DeviceModel(row[dt_key].decode())
            else:
                device_model = DeviceModel.YANDEX_STATION_PRO

            if device_model == DeviceModel.YANDEX_STATION_PRO and row[b'hdcp_14_key_wrapped']:
                data.append(cls.convert_data(row, device_model, process_keyboxes=True))
            elif device_model in keybox_free_models:
                data.append(cls.convert_data(row, device_model, process_keyboxes=False))

        return data

    def make_requests(self):
        data = self.read_data(self.Parameters.yt_path)

        for i, chunk in enumerate(self.split_into_parts(data, 5000), start=1):
            r = requests.post(self.upload_endpoint, json=chunk)
            if not r.ok:
                msg = 'Jangles request failed: status={}, content={}, chunk={}'.format(r.status_code, r.content, i)
                raise Exception(msg)

            logging.info('success for chunk %i', i)

    def on_execute(self):
        self.init_yt()
        self.make_local_jangles_requests(self.Parameters)
