"""
Various utilities for quasar build tools
"""
import logging
import os
import re
import shutil
import subprocess
import tarfile
import time
import zlib

from uuid import uuid4
from base64 import b64decode
from datetime import datetime
from contextlib import contextmanager
from inspect import getmembers
from urlparse import urlparse

import requests

import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm

from sandbox import common
from sandbox import sdk2
from sandbox.common import enum
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.environments import PipEnvironment
from sandbox.projects.common.solomon import push_to_solomon_v2
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.vcs import svn


class VCS(enum.Enum):
    enum.Enum.preserve_order()

    GIT = None
    SVN = None


class VCSSelector(sdk2.parameters.String):
    choices = [(_, _) for _ in VCS]
    default_value = VCS.GIT


class DefaultToLastResourceMixin(object):
    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=ctr.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastResource(DefaultToLastResourceMixin, sdk2.parameters.Resource):
    pass


class LastStableResource(LastResource):
    attrs = {"released": ctt.ReleaseStatus.STABLE}


class LastStableContainer(DefaultToLastResourceMixin, sdk2.parameters.Container):
    attrs = {"released": ctt.ReleaseStatus.STABLE}


def LastResourceWithAttrs(*a, **kw):
    """
    Convenience helper to create resource param classes on-the-fly.

    Use it like:

        ...
        class Parameters(sdk2.Task.Parameters):
            ...
            my_resource = LastResourceWithAttrs("Some Description", resource_type=MyResClass, attrs=dict(foo=123, released=ctt.ReleaseStatus.STABLE))
            ...
    """
    resource_attrs = kw.pop('attrs', {})

    class ParticularLastResource(LastResource):
        attrs = resource_attrs

    return ParticularLastResource(*a, **kw)


class SafePublishingMixing(object):
    """
    A mixin for `sdk2.Task` children to publish resources safely
    """

    def publish_safely(self, resources, comment='', copy=False):
        """
        Publishes resource.

        To make sure that they are not removed later the resource is first copied
        to some "safe" dir in the task home.

        :param dict resources:
            dict {resource_class: 'path/to/resorces'} to be published
        :param str comment:
            a comment to be added to the resource published
        :param bool copy:
            if resources should be copied instead of moving. Copying is slower, but
            crucial if you need resources later in the build in the original place.

        :returns: {resource_class: <resource instance>} map of published resources.
        """

        # NB: should never every remove this directory!
        res_dir = str(self.path('output_resources'))

        if not os.path.exists(res_dir):
            os.makedirs(res_dir)

        return {
            clazz: self._publish_safely_one(path, clazz, comment, res_dir, copy=copy)
            for clazz, path in resources.items()
        }

    def _publish_safely_one(self, path, clazz, comment, output_dir, copy=False):
        logging.info('Publishing %s as a resource %s' % (path, clazz))

        # create unique suffix for each resource
        # so resources with same name can be published
        suffix = str(uuid4())

        resource_dir = os.path.join(output_dir, suffix)
        # resource_dir should not exist yet
        os.makedirs(resource_dir)

        if copy:
            # copy resources to safety
            if os.path.isdir(path):
                logging.debug('Copying dir %s' % path)

                shutil.copytree(path, os.path.join(resource_dir, os.path.basename(path.rstrip('/'))))
            else:
                logging.debug('Copying file %s' % path)

                shutil.copy2(path, resource_dir)
        else:
            # move resource to safety
            logging.debug('Moving %s' % path)

            shutil.move(path, resource_dir)

        resource = clazz(
            self,
            comment,
            # publish resource copied to the safe dir
            os.path.join(resource_dir, os.path.basename(path.rstrip('/'))),
        )

        # workaround https://st.yandex-team.ru/SANDBOX-3751
        if hasattr(clazz, 'ttl'):
            resource.ttl = getattr(clazz, 'ttl')

        sdk2.ResourceData(resource).ready()

        return resource

    def untarball_dir_res(self, maybe_resource, dir_path, skip_missing=True):
        """
        Extracts contents of a tarball to `dir_path`, overwriting its contents.

        :param skip_missing: allow broken/missing resources. Won't throw NO_RES errors if resource retrieval fails.
        """
        dir_path = str(dir_path)  # maybe it's a sandboxy path

        if os.path.exists(dir_path):
            shutil.rmtree(dir_path)

        os.makedirs(dir_path)

        if maybe_resource:
            logging.info('Filling dir %s from res %s' % (dir_path, maybe_resource))

            try:
                RunHelper()('tar', 'xvzf', sdk2.ResourceData(maybe_resource).path, '-C', dir_path)
            except Exception as e:
                if skip_missing:
                    logging.warning('Failed to retrieve resource: %s', e)
                else:
                    raise
        else:
            logging.info('No resource given for %s -- leaving it empty' % dir_path)

    def tarball_publish_dir(self, resource_clazz, dir_path, **extra):
        """
        :param sdk2.Resource resource_clazz: to publish as
        :param str dir_path: where it is located
        :param extra: kwargs as in `publish_safely`

        Tarballs existing directory and publishes it as `resource_clazz`

        see https://wiki.yandex-team.ru/sandbox/faq/#pochemuzadachapadaetsoshibkojjsharingofresourcefailed
        """
        dir_path = str(dir_path)  # maybe it's a sandboxy path

        tarball = dir_path.rstrip('/') + '.tgz'

        RunHelper()('tar', 'cvzf', tarball, '-C', dir_path, '.')

        return self.publish_safely({resource_clazz: tarball}, **extra)[resource_clazz]


class Slack(object):
    """
    Namespace for slack-related functions
    NB: slack api is IPv4-only, use DnsType.DNS64
    """

    @staticmethod
    def notify(text=None, attachments=[], hook_url=None):
        """
        Send actual notication

        :param str hook_url: base url for app, see https://api.slack.com/incoming-webhooks. If not given falls back to one read from Vault
        :param str text: message text
        :param list attachments: attachments, see https://api.slack.com/docs/messages
        """
        if hook_url is None:
            hook_url = sdk2.Vault.data('quasar_ci_slack_notification_url')

        response = requests.post(
            hook_url,
            json=dict(
                text=text,
                attachments=attachments))

        # ensure it worked
        response.raise_for_status()

    @staticmethod
    def attachment(title, title_link=None, text=None, color=None):
        """
        Convenience builder for attachment
        :param str title:
        :param str title_link:
        :param str text: body text
        :param str color: color for marker in `FFFFFF`-format
        """
        return dict(title=title, title_link=title_link, text=text, color=color)


class RunHelper(object):
    """
    Helper to run commands

    :param env: for process
    :param base_dir: where to start cd'ing

    Usage ::

        r = RunHelper(env={....})

        with r.cd('foo'):
            r.run('foo', 'bar.txt')
    """

    def __init__(self, env={}, base_dir='.'):
        self._env = env
        self.base_dir = base_dir
        self._run_num = 0
        self._path = []  # dir elements

    def __call__(self, *command):
        self._run_num += 1

        return run_process(
            map(str, command),
            environment=self._env,
            work_dir=self.pwd(),
            log_prefix=(str(self._run_num) + '_' + '_'.join(map(str, command)).replace('/', '.'))[:32],  # limit log prefix to a meaningful value
        )

    def pwd(self):
        return os.path.join(self.base_dir, *self._path)

    @contextmanager
    def cd(self, *parts):
        for p in parts:
            self._path.append(p)

        try:
            yield self
        finally:
            for _ in parts:
                self._path.pop()

    @contextmanager
    def env(self, **extra_env):
        """
        Contextmanager to update env for the context
        """
        original = self._env

        try:
            self._env = dict(self._env.items() + extra_env.items())
            yield self
        finally:
            self._env = original


class ListableEnum(object):
    """
    A fancy mixin to add `X.all` to your enum-like classes

    >>> class MyEnum(ListableEnum):
    ...     FOO = 'foo'
    ...     BAR = 'bar'
    >>> 'foo' in MyEnum.all()
    True
    >>> len(MyEnum.all())
    2
    """
    @classmethod
    def all(cls):
        return [v for (n, v) in getmembers(cls) if n.upper() == n]

    @classmethod
    def named_all(cls):
        return [(n, v) for (n, v) in getmembers(cls) if n.upper() == n]

    @classmethod
    def for_name(cls, name):
        return dict(cls.named_all()).get(name)


class SignerMixin(object):
    """
    A mixin for signing someting as part of build process

    NB: add `SignerMixin.environments` to your task environments!
    """
    environments = [
        PipEnvironment('yandex-signer-client', '0.9.1', use_wheel=True),
    ]

    _run_helper = None

    # known certs for our use
    class Certs(ListableEnum):
        APK = 1
        TARGET_FILES = 177

    @property
    def run(self):
        if self._run_helper is None:
            self._run_helper = RunHelper()

        return self._run_helper

    def sign(self, path, cert_id):
        """
        :param path: to the file to sign
        :param cert_id: from the signer to use, see `SignerMixin.Certs`

        :returns: path to signed item
        """
        logging.info('Signing %s ...' % path)

        signed_path = self.path("signed_%s_%s" % (uuid4(), os.path.basename(str(path))))

        for i in range(3):  # three retries
            try:
                self.run(
                    'ya-signer',
                    '--request_timeout', '1200',  # see https://st.yandex-team.ru/QUASAR-9464
                    '--token', sdk2.Vault.data('robot-quasar-signer-oauth-token'),
                    '--cert_id', cert_id,
                    '--log', 'DEBUG',
                    '--infile', path,
                    '--outfile', signed_path,
                )
                break
            except Exception:
                if i < 2:
                    logging.exception('Failed to sign, retrying...')
                else:
                    raise

        logging.info('Done, result is at %s ' % signed_path)

        return str(signed_path)


class YAVExportMixin(object):
    """
    A mixin to export secrets from YAV

    Uses an ssh key stored in sandbox' Vault to access YAV
    Key and owner are made for Quasar, but are overridable via YAV_KEY_VAULT_OWNER / YAV_SSH_PRIVATE_KEY_VAULT_NAME / YAV_SSH_LOGIN properties.

    Class relies on ramdisk and will only export data there -- make sure your task has a ramdisk defined on it's own
    or inherits it's Requirements from convenience class added here.

    FIXME: when https://st.yandex-team.ru/SANDBOX-5877 is fixed use native mechanism!
    """

    YAV_KEY_VAULT_OWNER = 'QUASAR'
    YAV_SSH_PRIVATE_KEY_VAULT_NAME = 'robot-quasar-ssh-private-key'
    YAV_SSH_LOGIN = 'robot-quasar'

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 128, None)  # 128 Mb ramdrive for secrets

    @property
    def yatool(self):
        """
        A path to fresh checked out yatool
        """
        if not hasattr(self, '_yatool'):
            # export ya tool
            sdk2.svn.Arcadia.export(os.path.join(sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, "ya"), str(self.path()))

            self._yatool = str(self.path('ya'))

        return self._yatool

    @contextmanager
    def pkey(self):
        """
        A context manager to export private key and return path to private key file
        """
        try:
            private_key_path = os.path.join(str(self.ramdrive.path), 'pkey')

            with open(private_key_path, 'wb') as pk:
                pk.write(sdk2.Vault.data(self.YAV_KEY_VAULT_OWNER, self.YAV_SSH_PRIVATE_KEY_VAULT_NAME))

            yield private_key_path
        finally:
            os.remove(private_key_path)

    def yav_export(self, secret_or_version, *keys):
        """
        Exports keys from given yav secret version to a set of files in a directory in this task's ramdisk.
        Directory is created with a unique name.
        Assumes keys are not empty and checs each file.

        NB: keys contents should be python-printable (so no tarballs / zips per se) -- see https://st.yandex-team.ru/VAULT-163

        :param str secret_or_versionn: to take -- like sec-123123123123 or ver-123123123123. For secrets, gets last version
        :param list(str) keys: to export
        :returns: a path to the directory with exported files,
        """

        result_dir = os.path.join(str(self.ramdrive.path), str(uuid4()))

        os.makedirs(result_dir)

        with self.pkey() as private_key_path:
            for key in keys:
                with open(os.path.join(result_dir, key), 'wb') as out:
                    subprocess.check_call(
                        [
                            self.yatool,
                            'vault', 'get', 'version',
                            '--rsa-login', self.YAV_SSH_LOGIN,
                            '--rsa-private-key', private_key_path,
                            secret_or_version,
                            '-o', key,
                        ],
                        stdout=out,
                    )

                size = os.stat(os.path.join(result_dir, key)).st_size

                if not size:
                    raise IOError('Failed to extract key %s -- file is emty!' % key)
                else:
                    logging.debug('Exported key <%s> to <%s>, total %s bytes' % (key, os.path.join(result_dir, key), size))

        return result_dir

    def yav_export_b64_file(self, secret_or_version, key):
        """
        Utility method to export a base64-encoded file from yav secret key.
        # FIXME: we store tarball base64-enoded due to https://st.yandex-team.ru/VAULT-163, discard that when it's fixed!

        :param secret_or_version: to get file from from
        :param key: to export

        :returns: path to the file with key contents
        """

        base64_file_path = os.path.join(self.yav_export(secret_or_version, key), key)
        file_path = base64_file_path.rstrip('.base64') + '.unpacked'

        with open(file_path, 'wb') as dest:
            with open(base64_file_path) as src:
                dest.write(b64decode(src.read()))

        return file_path

    def yav_export_file(self, secret_or_version, key):
        """
        :type secret_or_version: str
        :param str key: name of file to export
        :return str: path to exported file
        """
        return os.path.join(self.yav_export(secret_or_version, key), key)

    def yav_export_secure_tar(self, tar_path, secure_secret_or_version, secure_key):
        """
        :param str tar_path: path to secured tar archive
        :param str secure_secret_or_version: yav secret or version with secure key
        :param str secure_key: name of secure key
        :return str: path to un
        """
        key = self.yav_export_file(secure_secret_or_version, secure_key)

        abs_tar_path = os.path.join(self.checkout_path, tar_path)
        key_link = os.path.splitext(abs_tar_path)[0]
        dec_tarball = key_link + '.tar'
        enc_tarball = dec_tarball + '.enc'
        subprocess.check_call(
            'openssl enc -d -in {} -out {} -aes-256-cbc -pbkdf2 -kfile {}'.format(enc_tarball, dec_tarball, key),
            shell=True)
        with tarfile.open(dec_tarball) as k:
            k.extractall(key_link)
        return key_link


class QuasarDBMixin(object):
    """
    This class provides convenience methods to access quasar's PostgreSQL databases.

    NB: to use it task has to include this class' Requirements!
    NB: resulting task only actually runnable by QUASAR group since vault items are hard-coded for now
    """

    class DBs(ListableEnum):
        """
        Known databases.
        Value is vault name for connuri.
        Connuri should follow format described for URIs in https://www.postgresql.org/docs/current/libpq-connect.html#LIBPQ-CONNSTRING
        """

        BACKEND = 'quasar-backend-connuri'
        BACKEND_RO = 'quasar-backend-ro-connuri'
        QUASMODROM = 'quasar-quasmodrom-connuri'
        QUASMODROM_RO = 'quasar-quasmodrom-ro-connuri'

    environments = [
        PipEnvironment("psycopg2-binary")
    ]

    @staticmethod
    def prepare_certs():
        run_process(
            ["curl", "https://crls.yandex.net/allCAs.pem", "-o", "allCAs.pem"],
            work_dir=".",
            log_prefix="curl"
        )

    @staticmethod
    def rearrange_connection_uri(connection_uri):
        """
        :param str connection_uri: like "postgresql://other@localhost/otherdb?connect_timeout=10&application_name=myapp"
        :return: connuri with hosts rearranged via rearrange_db_hosts
        """
        hosts = urlparse(connection_uri).hostname

        return connection_uri.replace(hosts, QuasarDBMixin.rearrange_db_hosts(hosts))

    def get_db_connection(self, connuri_secret_name):
        QuasarDBMixin.prepare_certs()  # wouldn't hurt

        connuri = sdk2.Vault.data("QUASAR", connuri_secret_name)

        import psycopg2
        return psycopg2.connect(QuasarDBMixin.rearrange_connection_uri(connuri))

    @staticmethod
    def rearrange_db_hosts(hosts):
        """
        Should use closest db host. Or any
        :param hosts: hosts string aka 'sas-host1.yandex.net,man-host2.yandex.net'
        :return:
        """
        dc = common.config.Registry().this.dc
        tail = []
        head = []
        for host in hosts.split(','):
            if host.startswith(dc):
                head.append(host)
            else:
                tail.append(host)

        return ",".join(head + tail)


def chunks(l, n):
    """Yield successive n-sized chunks from l."""
    for i in range(0, len(l), n):
        yield l[i:i + n]


class GitRefsMixin(object):
    class GitRefTypes(ListableEnum):
        Branches = 'heads'
        Tags = 'tags'

    def _git_ls_remote(self, git_url, what=None):
        all_refs = True
        if what in self.GitRefTypes.all():
            all_refs = False

        command = [
            'git',
            'ls-remote',
        ]
        if not all_refs:
            command += '--' + what
        command += git_url

        with ssh.Key(self, self.VAULT_OWNER, self.SSH_PRIVATE_KEY_VAULT_NAME):
            output = subprocess.Popen(
                command,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
            ).communicate()[0].strip('\n').split('\n')

        if not all_refs:
            ref_parts = 3
        else:
            ref_parts = 2

        if len(output) > 1:
            output = [
                # Reducing output to name of reference
                re.split(
                    r'/',
                    re.split(r'\s+', l)[1],
                    maxsplit=ref_parts - 1
                )[ref_parts - 1]

                for l in output
            ]

        return output

    def git_branches(self, git_url):
        return self._git_ls_remote(git_url, self.GitRefTypes.Branches)

    def git_tags(self, git_url):
        return self._git_ls_remote(git_url, self.GitRefTypes.Tags)


def check_ota_size(ota_resource, size_limit_mb):
    if size_limit_mb is None:
        logging.info("Not checking size limit info for resource because size limit is None")
        return

    size_limit = size_limit_mb * 1024 * 1024
    if ota_resource.size > size_limit:
        raise TaskFailure('OTA too big: {} > {} limit! {} bytes above limit.'.format(
            ota_resource.size,
            size_limit,
            ota_resource.size - size_limit
        ))


def get_disk_size(path):
    """
    :param str or sdk2.Path path:
    """
    return int(subprocess.check_output([
        'du', '-sb', str(path),
    ]).split()[0])


def get_quasar_sizes(path):
    """
    Calculate disk space used for each nonlink item of quasar daemons directory recursively.
    :param str or sdk2.Path path: path to quasar daemons directory.
    :return: dict relative_path -> disk_usage.
    """
    path = str(path)
    sizes_dict = {}
    for root, _, files in os.walk(path):
        for name in files:
            file_path = os.path.join(root, name)
            if not os.path.islink(file_path):
                sizes_dict[file_path[len(path) + 1:]] = os.path.getsize(file_path)

    return sizes_dict


def push_sensors_to_solomon(service, sensors):
    """
    Mindly general function to send quasar'ish sandbox sensors to solomon

    :param str service: solomon service name, e.g. 'build_stats' or 'device_group_info' -- logical sensor group
    :param list sensors: list of per-sensor dicts like ::

        [
            {
                "labels": {
                        "sensor": "<sensor_name>",  # kinda-mandatory
                        # any extra labels for filtering
                    },
                "value": 123.45,
                "ts": "2019-02-03T04:11:43.000000Z"  # optional time in ISO-8601 format
            },
        ]

    If ts is not given it is filled with current ts on solomon side -- see https://wiki.yandex-team.ru/solomon/api/push/
    """
    solomon_token = sdk2.Vault.data('QUASAR', 'solomon_token')

    solomon_params = {
        'project': 'quasar',
        'cluster': 'sandbox',
        'service': service,
    }

    push_to_solomon_v2(token=solomon_token, params=solomon_params, sensors=sensors)


def push_sizes_to_solomon(binaries_resource, quasar_sizes, ota_image_resource, apk_size=None):
    if not binaries_resource.svn_revision:
        logging.info('No commit_time found for quasar daemons. Not pushing to solomon.')
        return

    commit_time = svn.Arcadia.info('arcadia:/arc/trunk/arcadia@' + binaries_resource.svn_revision)['date']
    timestamp = (
        time.mktime(datetime.strptime(commit_time, '%Y-%m-%dT%H:%M:%S.%fZ').timetuple()) -
        time.altzone)

    solomon_sensors = [
        {
            'labels': {'sensor': 'size_ota_image', 'platform': binaries_resource.quasar_platform},
            'ts': timestamp,
            'value': float(ota_image_resource.size),
        },
        {
            'labels': {'sensor': 'count_quasar_daemons_files', 'platform': binaries_resource.quasar_platform},
            'ts': timestamp,
            'value': float(len(quasar_sizes)),
        },
    ]
    for item, item_size in quasar_sizes.items():
        solomon_sensors.append({
            'labels': {'sensor': 'size_quasar_daemons_{}'.format(item), 'platform': binaries_resource.quasar_platform},
            'ts': timestamp,
            'value': float(item_size),
        })
    if apk_size is not None:
        solomon_sensors.append({
            'labels': {'sensor': 'size_quasar_apk', 'platform': binaries_resource.quasar_platform},
            'ts': timestamp,
            'value': float(apk_size),
        })

    push_sensors_to_solomon(service='build_stats', sensors=solomon_sensors)


def get_crc_32(file_path):
    def my_crc32(crc32):
        """
        simulates cmdline tool crc32 -- returns hex of uint32 of crc32-checksum with leading 0x removed
        """
        # zlib's crc32 removes int, convert to uint
        if crc32 < 0:
            crc32 = 2 ** 32 + crc32

        return hex(crc32)[2:]  # trim leading 0x

    with open(file_path, 'rb') as opened_file:
        crc32 = 0
        while True:
            data = opened_file.read((1 << 31) - 1)
            if not data:
                break
            crc32 = zlib.crc32(data, crc32)
    return my_crc32(crc32)


def get_resource_crc32(resource):
    path = str(sdk2.ResourceData(resource).path)
    return get_crc_32(path)


def create_task_logger(task, log_path, logger_name):
    """
    :param sdk2.Task task:
    :param str log_path:
    :param str logger_name:
    :return task logger with name `logger_name` and FileHandler writing to `log_path`:
    """
    log_file = task.log_path(log_path)
    logging.debug('Write %s logs to %s', logger_name, log_file)

    logger = logging.getLogger(logger_name)
    logger.handlers = [
        h for h in logger.handlers if not isinstance(h, logging.FileHandler)
    ]
    logger.addHandler(logging.FileHandler(str(log_file)))
    return logger
