import logging
import os
import multiprocessing
import urlparse
import subprocess

import sandbox.common as sandbox_common
import sandbox.sandboxsdk.errors as sandbox_errors

from sandbox import sdk2
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.process import run_process


class NoFailSubprocessException(sandbox_errors.SubprocessErrorBase):
    pass


class VCS(sandbox_common.enum.Enum):
    sandbox_common.enum.Enum.lower_case()

    GIT = None
    REPO = None


class Containers(object):
    REPO_CONTAINER_ID = 1872629987


class CheckoutMixinBase(object):
    def __init__(self):
        self.repository_url = None
        self.repository_tag = None
        self.repository_user = None
        self.checkout_path = None
        self.ssh_private_key_vault_name = None
        self.ssh_private_key_vault_owner = None

    @property
    def checkout_url(self):
        """
        :rtype: str
        """
        split = urlparse.urlsplit(self.repository_url)
        if self.repository_user:
            netloc = split.netloc.split('@')[-1]
            split = split._replace(netloc='{}@{}'.format(self.repository_user, netloc))

        return split.geturl()

    @staticmethod
    def safe_run_command(*args, **kwargs):
        try:
            subprocess.check_call(*args, **kwargs)
        except subprocess.CalledProcessError as e:
            raise sandbox_common.errors.TaskError('Command exited with code {}, {}'.format(e.returncode, e.message))


class GitCheckout(object):
    @staticmethod
    def do(
            sandbox_task,
            checkout_url,
            repository_tag,
            ssh_private_key_vault_owner,
            ssh_private_key_vault_name,
            checkout_path,
            pull_lfs=False,
            clone_depth=1,
    ):
        logging.info("Checking out code with git from '%s' at tag '%s'...", checkout_url, repository_tag)

        with ssh.Key(
            sandbox_task,
            ssh_private_key_vault_owner,
            ssh_private_key_vault_name
        ):
            # shallow clone to reduce data transfer
            run_process(
                [
                    'git',
                    'clone',
                    '--depth', str(clone_depth),
                    '--branch', repository_tag,
                    checkout_url,
                    checkout_path
                ],
                log_prefix='git_clone',
            )

            if pull_lfs:
                run_process(
                    'git lfs pull',
                    work_dir=checkout_path,
                    log_prefix='git_lfs_pull',
                )


class GitCheckoutMixin(CheckoutMixinBase):
    class GitParameters(object):
        clone_depth = 1
        pull_lfs = False

    def git_checkout(self):
        GitCheckout.do(
            sandbox_task=self,
            checkout_url=self.checkout_url,
            repository_tag=self.repository_tag,
            ssh_private_key_vault_owner=self.ssh_private_key_vault_owner,
            ssh_private_key_vault_name=self.ssh_private_key_vault_name,
            checkout_path=self.checkout_path,
            pull_lfs=self.GitParameters.pull_lfs,
            clone_depth=self.GitParameters.clone_depth,
        )


class RepoCheckoutMixin(CheckoutMixinBase):
    class RepoParameters(object):
        init_depth = 1
        current_branch = True
        cpu_count = multiprocessing.cpu_count()

    @property
    def repo_overlay_inspections(self):
        """
        :return: {path: git_tag}
        :rtype: Dict[str, str]
        """
        return dict()

    def repo_checkout(self):
        clone_path = sdk2.path.Path(self.checkout_path)
        clone_path.mkdir(parents=True, exist_ok=True)

        logging.info("Checking out code with repo from '%s' at tag '%s'...", self.checkout_url, self.repository_tag)

        with ssh.Key(
            self, self.ssh_private_key_vault_owner,
            self.ssh_private_key_vault_name
        ):
            init_args = [
                'repo',
                'init',
                '-u', self.checkout_url,
                '-b', self.repository_tag,
            ]

            if self.RepoParameters.init_depth is not None:
                init_args += ['--depth', str(self.RepoParameters.init_depth)]

            if self.RepoParameters.current_branch:
                init_args += ['--current-branch']

            # Init repo with manifest
            run_process(
                init_args,
                work_dir=self.checkout_path,
                log_prefix='repo_init',
                shell=True,
                exc_class=NoFailSubprocessException,
            )

            repo_sync_base_list = [
                'repo',
                'sync',
                '-c',
            ]

            if self.repo_custom_manifest_path:
                if os.path.exists(self.repo_custom_manifest_path):
                    repo_sync_base_list.append('-m')
                    repo_sync_base_list.append(self.repo_custom_manifest_path)
                else:
                    raise sandbox_common.errors.TaskFailure(
                        'Custom manifest file does not exist.\n{}'.format(self.repo_custom_manifest_path))

            # Sync projects, fetch from remotes only.
            repo_sync_network_list = repo_sync_base_list + [
                '-n',
                '-j' + str(self.repository_checkout_threads)
            ]
            run_process(
                repo_sync_network_list,
                work_dir=self.checkout_path,
                log_prefix='repo_sync',
                exc_class=NoFailSubprocessException,
            )

            # Sync projects, update working tree only.
            repo_sync_local_list = repo_sync_base_list + [
                '-l',
                '-j' + str(self.RepoParameters.cpu_count)
            ]

            run_process(
                repo_sync_local_list,
                work_dir=self.checkout_path,
                log_prefix='repo_sync',
            )

            for inspection in self.repo_overlay_inspections:
                run_process(
                    [
                        'repo',
                        'download',
                        self.repo_overlay_inspections[inspection],
                        inspection
                    ],
                    work_dir=self.checkout_path,
                    log_prefix='repo_download',
                    exc_class=NoFailSubprocessException,
                )
