import datetime
import logging
import requests
from time import sleep

from sandbox import sdk2
from sandbox.common import errors

TOKEN_SECRET_NAME = 'quasar_web_view_push_sender_token'
EVENT = 'web'
PUSH_URL = 'https://push.yandex.ru/v2/send'
DEFAULT_YABRO_TYPE = 'regular'
DEFAULT_OVERLAY_STYLE = 'opaque'
DEFAULT_OVERLAY_STATE = 'collapsed'
DEFAULT_OVERLAY_ADJUSTMENT = 'pan'
SITES = ['https://v134d1.wdevx.ya.ru?exp=rum_info',
         'https://st-v134d1.wdevx.yandex.net/tmpl/station/simple.html',
         'https://st-v134d1.wdevx.yandex.net/tmpl/station/animations.html',
         'https://v134d1.wdevx.yandex.ru/?content=tv&exp=rum_info',
         'https://v134d1.wdevx.yandex.ru/?exp=rum_info',
         'https://v134d1.wdevx.yandex.ru/efir?exp=rum_info']


class QuasarWebViewPushSender(sdk2.Task):
    """
    A task that sends pushes to specified users in order to open specified urls inside WebView
    with given timeout for some duration
    """
    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 8  # 8 hours

        task_duration = sdk2.parameters.Integer("Task duration in hours", default=6, required=True)
        iterations_timeout = sdk2.parameters.Integer("Timeout between opening sites in seconds", default=30,
                                                     required=True)
        user_uids = sdk2.parameters.String("User uids", required=True)
        sites = sdk2.parameters.String("Sites", default=', '.join(SITES), required=True)
        push_url = sdk2.parameters.String("Push url", default=PUSH_URL, required=True)
        yabro_type = sdk2.parameters.String("Yabro type (surface | regular)", default=DEFAULT_YABRO_TYPE, required=True)
        overlay_style = sdk2.parameters.String("Overlay style (transparent | opaque)",
                                               default=DEFAULT_OVERLAY_STYLE, required=True)
        overlay_state = sdk2.parameters.String("Overlay state (hidden | collapsed | expanded | anim)",
                                               default=DEFAULT_OVERLAY_STATE, required=True)
        overlay_adjustment = sdk2.parameters.String("Overlay adjustment (pan | fit)",
                                                    default=DEFAULT_OVERLAY_ADJUSTMENT, required=True)

    def on_execute(self):
        try:
            token = sdk2.Vault.data(TOKEN_SECRET_NAME)

        except errors.VaultError:
            raise errors.TaskFailure("Failed to read token '%s' from Vault" % TOKEN_SECRET_NAME)

        headers = {'Content-Type': 'application/json;charset=UTF-8'}
        duration = datetime.timedelta(hours=self.Parameters.task_duration)
        start_time = datetime.datetime.now()

        while datetime.datetime.now() - start_time < duration:
            for site in self.Parameters.sites.split(','):
                for user in self.Parameters.user_uids.split(','):
                    payload = {'token': token, 'user': user.strip(), 'event': EVENT}
                    r = requests.post(
                        url=self.Parameters.push_url,
                        data=None,
                        json={'payload': {'data': site.strip(),
                                          'yabro': self.Parameters.yabro_type,
                                          'style': self.Parameters.overlay_style,
                                          'state': self.Parameters.overlay_state,
                                          'adjustment': self.Parameters.overlay_adjustment}},
                        headers=headers,
                        params=payload
                    )
                    logging.info('Request {user uid: %s; url: %s; status: %s}' % (user.strip(), site.strip(), r.text))
                sleep(self.Parameters.iterations_timeout)
