# -*- coding: utf-8 -*-
import os
import time
import logging
import json
from urllib2 import urlopen
from urllib import quote
from sandbox.sandboxsdk import parameters, environments


class YtProxy(parameters.SandboxStringParameter):
    name = 'yt_proxy'
    description = 'YT cluster to use for storing data'
    default_value = 'banach'


class YtTablePath(parameters.SandboxStringParameter):
    name = 'yt_table_path'
    description = 'YT path to store table with querysearch data'
    default_value = '//home'
    required = True


class QuerysearchNamespace(parameters.SandboxStringParameter):
    name = 'qs_namespace'
    description = 'querysearch namespace, e.g. addrsnip:docid_setprops'
    required = True


class PrepareSaasQuerysearchData():

    environment = (
        environments.PipEnvironment('yandex-yt', use_wheel=True),
    )
    input_parameters = (YtProxy, YtTablePath, QuerysearchNamespace)

    def get_handle_host(self, is_testing):
        lld = 'fastsnips-testing' if is_testing else 'fastsnips'
        return '{}.ferryman.n.yandex-team.ru'.format(lld)

    def upload_to_yt(self, data, table_prefix, token, latest_link=None):
        """
        Expects `data' to be a generator that yields dicts

        Common dict keys (that become columns):
            Subkey_Url
            Subkey_ZDocId
            Subkey_UserRegion
            Data_JSON
            Data_TSKV

        All possible keys are listed here:
            kernel/querydata/saas_yt/idl/qd_saas_input_record.proto
        """
        import yt.wrapper as yt

        cluster = self.ctx[YtProxy.name]
        if not cluster.endswith('.yt.yandex.net'):
            cluster += '.yt.yandex.net'

        os.environ['YT_PROXY'] = cluster
        client = yt.YtClient(proxy=cluster, token=token)
        ts = int(time.time())
        table_name = self.ctx[YtTablePath.name] + '/{}.{}'.format(table_prefix, ts)
        self.ctx['YtTableFullPath'] = table_name
        logging.info("Will write to table {} on {}".format(table_name, cluster))
        table = yt.TablePath(table_name, client=client)
        client.write_table(table, data, format=yt.JsonFormat(attributes={"encode_utf8": False}))

        if latest_link:
            link_name = self.ctx[YtTablePath.name] + '/' + latest_link
            client.link(table_name, link_name, force=True)

    def on_release(self, is_testing=False):
        """ pull handle to trigger data update """
        query = json.dumps([{
            "Path": self.ctx['YtTableFullPath'],
            "Namespace": self.ctx[QuerysearchNamespace.name],
            "Cluster": self.ctx[YtProxy.name],
            "Timestamp": int(time.time()) * 1000000
        }])
        request_str = "http://{}/add-full-tables?tables={}".format(self.get_handle_host(is_testing), quote(query))
        logging.info(request_str)
        logging.info(urlopen(request_str).read())
