import json
import subprocess
import time
import urlparse
from distutils.version import LooseVersion

import requests
from six.moves import http_client

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.qyp.CopyVmImage import CopyVmImage
from sandbox.projects.qyp.MergeQemuImage import MergeQemuImage


MIN_VMAGENT_VERSION = LooseVersion('0.18')
VMPROXY_URL_MAP = {
    'SAS': 'https://vmproxy.sas-swat.yandex-team.ru/',
    'MAN': 'https://vmproxy.man-swat.yandex-team.ru/',
    'VLA': 'https://vmproxy.vla-swat.yandex-team.ru/',
    'MYT': 'https://vmproxy.myt-swat.yandex-team.ru/',
    'IVA': 'https://vmproxy.iva-swat.yandex-team.ru/',
    'TEST_SAS': 'https://dev-vmproxy.n.yandex-team.ru/',
    'MAN_PRE': 'https://dev-vmproxy.n.yandex-team.ru/',
}
TAGS_MAP = {
    'SAS': Tag.SAS,
    'MAN': Tag.MAN,
    'VLA': Tag.VLA,
    'MYT': Tag.MYT,
    'IVA': Tag.IVA,
    'TEST_SAS': Tag.SAS,
    'MAN_PRE': Tag.MAN,
}
CLUSTER_CHOICES = [(x, x) for x in VMPROXY_URL_MAP.iterkeys()]

DOWNLOAD_SPEED = 40  # Mb/s minimal speed within one dc
WAIT_SHARE_TICK = 30


def calculate_time(size_mb, download_speed):
    return size_mb / (download_speed / 8)


class VmproxyClient(object):
    def __init__(self, cluster, token):
        self._base_url = VMPROXY_URL_MAP.get(cluster)
        self._session = requests.Session()
        self._session.headers['Authorization'] = 'OAuth {}'.format(token)
        self._session.headers['Content-Type'] = 'application/json'

    def post(self, url, data):
        deadline = time.time() + 600
        attempt = 0

        while True:
            attempt += 1
            resp = self._session.post(urlparse.urljoin(self._base_url, url), data=json.dumps(data))
            if http_client.BAD_REQUEST <= resp.status_code < http_client.INTERNAL_SERVER_ERROR:
                msg = 'Vmproxy request error: {} {}'.format(resp.status_code, resp.text)
                raise common.errors.TaskError(msg)

            if time.time() > deadline:
                resp.raise_for_status()
            else:
                if resp.status_code >= 500 and resp.status_code < 600:
                    time.sleep(min(attempt * 2, 30))
                    continue
                else:
                    resp.raise_for_status()

            break

        return resp.json()


class BackupQemuVm(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.Group.LINUX

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400  # 24h
        cluster = sdk2.parameters.String('YP Cluster', required=True, default='TEST_SAS',
                                         choices=CLUSTER_CHOICES)
        vm_id = sdk2.parameters.String('Virtual Machine Id', required=True)
        storage = sdk2.parameters.String('Storage for backup image file', required=True, default='SANDBOX_RESOURCE',
                                         choices=[
                                             ('Sandbox', 'SANDBOX_RESOURCE'),
                                             ('QDM', 'QDM')
                                         ])
        with sdk2.parameters.Output:
            result_url = sdk2.parameters.String('Qemu backup image url')

    def _get_vmproxy_client(self):
        return VmproxyClient(
            cluster=str(self.Parameters.cluster),
            token=sdk2.Vault.data('QYP', 'ROBOT_VMPROXY_TOKEN'),
        )

    def _get_qdm_tvm_ticket(self):
        import ticket_parser2
        import ticket_parser2.api.v1

        src = 2010666
        dst = 2015617  # qdm server

        secret = sdk2.Vault.data('QYP', 'QYP_TVM_QNOTIFIER_SECRET')
        ts = int(time.time())

        tvm_keys = requests.get(
            'https://tvm-api.yandex.net/2/keys?lib_version=%s' % (ticket_parser2.__version__, )
        ).content

        svc_ctx = ticket_parser2.api.v1.ServiceContext(src, secret, tvm_keys)

        ticket_response = requests.post(
            'https://tvm-api.yandex.net/2/ticket/',
            data={
                'grant_type': 'client_credentials',
                'src': src, 'dst': dst, 'ts': ts,
                'sign': svc_ctx.sign(ts, dst)
            }
        ).json()

        ticket_for_dst = ticket_response[str(dst)]['ticket']
        return ticket_for_dst

    def _is_sandbox_storage(self):
        return self.Parameters.storage == 'SANDBOX_RESOURCE'

    def _is_qdm_storage(self):
        return self.Parameters.storage == 'QDM'

    def on_execute(self):
        client = self._get_vmproxy_client()
        if not self.Context.backup_id:
            self.set_info('Get VM information')
            vm_info = client.post('/api/GetVm/', {
                'vm_id': self.Parameters.vm_id
            })['vm']
            self._check_vmagent_version(vm_info)
            self._make_yp_record(client, vm_info)
        if not self.Context.image_urls:
            self._get_image_urls(client)
        if not self.Parameters.result_url:
            if not self._is_sandbox_storage():
                # if is_qdm_storage will also be here -- just save qdm key
                self.Parameters.result_url = self.Context.image_urls['imageUrl']
            elif not self.Context.sub_task_id:
                self._make_resources()
            else:
                delta_url = self.Context.image_urls['deltaUrl']
                res_type = 'QEMU_MERGED_IMAGE' if delta_url else 'OTHER_RESOURCE'
                result_res = sdk2.Resource.find(sdk2.Resource[res_type], task_id=self.Context.sub_task_id).first()
                if result_res:
                    self.Parameters.result_url = result_res.skynet_id
                    self.set_info('Scheduled task done!')
                else:
                    raise common.errors.TaskError('Making backup resource failed, try again')

    def _check_vmagent_version(self, vm_info):
        version = vm_info['spec']['vmagentVersion']
        if version == 'N/A':
            version = LooseVersion('0')
        else:
            version = LooseVersion(version)
        if version < MIN_VMAGENT_VERSION:
            raise common.errors.TaskError("Minimun vmagent version {}, got {}".format(MIN_VMAGENT_VERSION, version))

    def _make_yp_record(self, client, vm_info):
        task_id = str(self.id)
        backup_list = client.post('/api/ListBackup/', {
            'vm_id': self.Parameters.vm_id
        })
        for backup in backup_list['backups']:
            if backup['spec']['sandboxTaskId'] == task_id:
                self.Context.backup_id = backup['meta']['id']
                break
        if self.Context.backup_id:
            # Created via vmproxy api
            client.post('/api/UpdateBackup/', {
                'id': self.Context.backup_id,
                'vm_id': self.Parameters.vm_id,
                'status': {
                    'state': 'IN_PROGRESS',
                }
            })
        else:
            # Created via sandbox scheduler
            backup = client.post('/api/CreateBackup/', {
                'spec': {
                    'type': self.Parameters.storage,
                    'vm_id': self.Parameters.vm_id,
                    'sandbox_task_id': task_id,
                    'generation': int(vm_info['status']['state']['generation']),
                }
            })
            self.Context.backup_id = backup['backup']['meta']['id']

    def _init_qdm_upload_session(self, vm_id, dc):
        response = requests.post(
            'http://qdm.yandex-team.ru/init_upload_session',
            headers={
                'X-Ya-Service-Ticket': self._get_qdm_tvm_ticket(),
            },
            json={
                'vm_id': vm_id,
                'dc': dc,
            }
        )

        response.raise_for_status()

        return response.json()['key']

    def _get_image_urls(self, client):
        if not self._is_qdm_storage():
            self.set_info('Sharing VM image')
        else:
            self.set_info('Uploading image to MDS')

        # Start backup process
        if not self._is_qdm_storage():
            client.post('/api/MakeAction/', {
                'vm_id': {
                    'pod_id': self.Parameters.vm_id,
                },
                'action': 'BACKUP',
            })
        else:
            qdm_key = self._init_qdm_upload_session(self.Parameters.vm_id, self.Parameters.cluster)

            client.post('/api/MakeAction/', {
                'vm_id': {
                    'pod_id': self.Parameters.vm_id
                },
                'action': 'QDMUPLOAD',
                'qdmreq': {
                    'key': qdm_key
                }
            })

        def checker():
            status = client.post('/api/GetStatus/', {
                'vm_id': {
                    'pod_id': self.Parameters.vm_id,
                },
            })
            correct_backup = False
            if status['state'].get('backup', False):
                if (
                    status['state']['backup'].get('imageUrl', False) or
                    status['state']['backup'].get('deltaUrl', False)
                ):
                    correct_backup = True

            return status['state'].get('backup') if correct_backup else None

        image_urls, _ = common.utils.progressive_waiter(
            tick=WAIT_SHARE_TICK,
            max_tick=WAIT_SHARE_TICK,
            max_wait=self.Parameters.kill_timeout,
            checker=checker
        )

        if not image_urls['imageUrl'] and not image_urls['deltaUrl']:
            raise common.errors.TaskError("Can't get image urls from vm status")

        self.Context.image_urls = image_urls

    def _get_images_size(self):
        image_urls = self.Context.image_urls
        resp = subprocess.check_output(['sky', 'files', '--json', image_urls['imageUrl']])
        size = json.loads(resp)[0]['size']
        delta_url = image_urls['deltaUrl']
        if delta_url:
            resp = subprocess.check_output(['sky', 'files', '--json', delta_url])
            size += json.loads(resp)[0]['size']
        return size / 1024 ** 2  # Mb

    def _make_resources(self):
        image_urls = self.Context.image_urls
        imgs_size = self._get_images_size()
        is_merge = None
        if image_urls['deltaUrl']:
            is_merge = True
            kill_timeout = max(calculate_time(imgs_size, DOWNLOAD_SPEED) * 2, 3 * 60 * 60)
            sub_task = MergeQemuImage(
                self,
                description='Merge qemu vm image',
                image_url=image_urls['imageUrl'],
                delta_url=image_urls['deltaUrl'],
            )
            sub_task.Requirements.disk_space = imgs_size * 2
            sub_task.Parameters.kill_timeout = kill_timeout
        else:
            is_merge = False
            kill_timeout = max(calculate_time(imgs_size, DOWNLOAD_SPEED), 3 * 60 * 60)
            sub_task = CopyVmImage(
                self,
                description='Copy qemu vm image',
                image_url=image_urls['imageUrl'],
            )
            sub_task.Requirements.disk_space = imgs_size + 1024
            sub_task.Parameters.kill_timeout = kill_timeout

        sub_task.Requirements.client_tags &= TAGS_MAP[str(self.Parameters.cluster)]
        sub_task.Parameters.priority.cls = ctt.Priority.Class.SERVICE
        sub_task.Parameters.priority.scls = ctt.Priority.Subclass.HIGH
        sub_task.save().enqueue()
        self.Context.sub_task_id = sub_task.id
        self.set_info(
            'Scheduled %s task: <a href="https://sandbox.yandex-team.ru/task/%d/view">%d</a>' % (
                'merge' if is_merge else 'copy', sub_task.id, sub_task.id
            ),
            do_escape=False
        )
        raise sdk2.WaitTask(sub_task.id, ctt.Status.Group.FINISH, wait_all=True)

    def on_failure(self, prev_status):
        if self.Context.backup_id:
            client = self._get_vmproxy_client()
            client.post('/api/RemoveBackup/', {
                'id': self.Context.backup_id,
                'vm_id': self.Parameters.vm_id,
            })
            self.Context.backup_id = ''
        self.set_info('Backup error occuped')
        super(BackupQemuVm, self).on_failure(prev_status)

    def on_success(self, prev_status):
        client = self._get_vmproxy_client()
        client.post('/api/UpdateBackup/', {
            'id': self.Context.backup_id,
            'vm_id': self.Parameters.vm_id,
            'status': {
                'state': 'COMPLETED',
                'url': self.Parameters.result_url,
            }
        })
        self.set_info('All done!')
        super(BackupQemuVm, self).on_success(prev_status)
