import os
import tarfile

from sandbox.projects.common.nanny import nanny
from sandbox.projects.nanny.common.build_helpers import git_clone_and_checkout
from sandbox.projects.resource_types import nanny_releasers
from sandbox.sandboxsdk.environments import VirtualEnvironment

from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process


class VmproxyCode(sdk2.Resource):
    releasable = True
    auto_backup = True
    releasers = nanny_releasers + ['max7255']


class VmproxyEnv(sdk2.Resource):
    releasable = True
    auto_backup = True
    releasers = nanny_releasers + ['max7255']


class BuildVmproxy(nanny.ReleaseToNannyTask2, sdk2.Task):
    code_path = 'vmproxy'
    req_file_name = 'pip-requirements.txt'
    dev_req_file_name = 'pip-dev-requirements.txt'
    tar_code_name = 'code.tar.gz'
    tar_venv_name = 'venv.tar.gz'
    GIT_URL = 'https://{username}:{password}@bitbucket.yandex-team.ru/scm/Nanny/nanny'

    class Parameters(sdk2.Parameters):
        ref_id = sdk2.parameters.String('Git ref id', default='master', required=True)
        ref_sha = sdk2.parameters.String('Git ref SHA', required=False)
        release = sdk2.parameters.Bool('Create resources', default=False)

    def _get_clone_url(self):
        oauth_token = sdk2.Vault.data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    def on_execute(self):
        package_path = str(self.path('nanny'))
        clone_url = self._get_clone_url()

        git_clone_and_checkout(
            package_url=clone_url,
            package_path=package_path,
            ref_id=self.Parameters.ref_sha or self.Parameters.ref_id
        )

        # Install all packages from wheels and test
        pwd = os.getcwd()
        try:
            abs_code_path = os.path.join(package_path, self.code_path)
            os.chdir(abs_code_path)  # for relative path in requirements
            req_path = os.path.join(abs_code_path, self.req_file_name)
            tar_venv_path = str(self.path(self.tar_venv_name))
            with VirtualEnvironment(str(self.path('env'))) as venv:
                venv.pip('pip==9.0.1')
                venv.pip('--no-index --find-links=./wheels -r {}'.format(req_path))
                venv.pack_bundle(tar_venv_path)
                self._run_tests(abs_code_path, venv)
        finally:
            os.chdir(pwd)

        if self.Parameters.release:
            # Make resources
            tar_code_path = str(self.path(self.tar_code_name))
            with tarfile.open(tar_code_path, 'w:gz') as tf:
                tf.add(abs_code_path, self.code_path)
            vmproxy_code = VmproxyCode(self, 'Vmproxy source code', tar_code_path, 'linux')
            sdk2.ResourceData(vmproxy_code).ready()

            vmproxy_env = VmproxyEnv(self, 'Virtual env', tar_venv_path, 'linux')
            sdk2.ResourceData(vmproxy_env).ready()

    def _run_tests(self, abs_code_path, venv):
        req_path = os.path.join(abs_code_path, self.dev_req_file_name)
        venv.pip('--no-index --find-links=./wheels -r {}'.format(req_path))

        run_process(
            ['./test.sh'],
            log_prefix='test_sh',
            work_dir=abs_code_path
        )
